"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.normalizeRuleExecutionStatsAggregationResult = exports.getRuleExecutionStatsAggregation = void 0;
var _lodash = require("lodash");
var _rule_monitoring = require("../../../../../../../../common/detection_engine/rule_monitoring");
var _es_aggregations = require("../../../utils/es_aggregations");
var f = _interopRequireWildcard(require("../../../event_log/event_log_fields"));
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getRuleExecutionStatsAggregation = aggregationContext => {
  return {
    totalExecutions: {
      cardinality: {
        field: f.RULE_EXECUTION_UUID
      }
    },
    executeEvents: {
      filter: {
        term: {
          [f.EVENT_ACTION]: 'execute'
        }
      },
      aggs: {
        executionDurationMs: {
          percentiles: {
            field: f.RULE_EXECUTION_TOTAL_DURATION_MS,
            missing: 0,
            percents: _es_aggregations.DEFAULT_PERCENTILES
          }
        },
        scheduleDelayNs: {
          percentiles: {
            field: f.RULE_EXECUTION_SCHEDULE_DELAY_NS,
            missing: 0,
            percents: _es_aggregations.DEFAULT_PERCENTILES
          }
        }
      }
    },
    statusChangeEvents: {
      filter: {
        bool: {
          filter: [{
            term: {
              [f.EVENT_ACTION]: _rule_monitoring.RuleExecutionEventType['status-change']
            }
          }],
          must_not: [{
            terms: {
              [f.RULE_EXECUTION_STATUS]: [_rule_monitoring.RuleExecutionStatus.running, _rule_monitoring.RuleExecutionStatus['going to run']]
            }
          }]
        }
      },
      aggs: {
        executionsByStatus: {
          terms: {
            field: f.RULE_EXECUTION_STATUS
          }
        }
      }
    },
    executionMetricsEvents: {
      filter: {
        term: {
          [f.EVENT_ACTION]: _rule_monitoring.RuleExecutionEventType['execution-metrics']
        }
      },
      aggs: {
        gaps: {
          filter: {
            exists: {
              field: f.RULE_EXECUTION_GAP_DURATION_S
            }
          },
          aggs: {
            totalGapDurationS: {
              sum: {
                field: f.RULE_EXECUTION_GAP_DURATION_S
              }
            }
          }
        },
        searchDurationMs: {
          percentiles: {
            field: f.RULE_EXECUTION_SEARCH_DURATION_MS,
            missing: 0,
            percents: _es_aggregations.DEFAULT_PERCENTILES
          }
        },
        indexingDurationMs: {
          percentiles: {
            field: f.RULE_EXECUTION_INDEXING_DURATION_MS,
            missing: 0,
            percents: _es_aggregations.DEFAULT_PERCENTILES
          }
        }
      }
    },
    messageContainingEvents: {
      filter: {
        terms: {
          [f.EVENT_ACTION]: [_rule_monitoring.RuleExecutionEventType['status-change'], _rule_monitoring.RuleExecutionEventType.message]
        }
      },
      aggs: {
        messagesByLogLevel: {
          terms: {
            field: f.LOG_LEVEL
          }
        },
        ...(aggregationContext === 'whole-interval' ? {
          errors: {
            filter: {
              term: {
                [f.LOG_LEVEL]: _rule_monitoring.LogLevel.error
              }
            },
            aggs: {
              topErrors: {
                categorize_text: {
                  field: 'message',
                  size: 5,
                  similarity_threshold: 99
                }
              }
            }
          },
          warnings: {
            filter: {
              term: {
                [f.LOG_LEVEL]: _rule_monitoring.LogLevel.warn
              }
            },
            aggs: {
              topWarnings: {
                categorize_text: {
                  field: 'message',
                  size: 5,
                  similarity_threshold: 99
                }
              }
            }
          }
        } : {})
      }
    }
  };
};
exports.getRuleExecutionStatsAggregation = getRuleExecutionStatsAggregation;
const normalizeRuleExecutionStatsAggregationResult = (aggregations, aggregationLevel) => {
  const totalExecutions = aggregations.totalExecutions || {};
  const executeEvents = aggregations.executeEvents || {};
  const statusChangeEvents = aggregations.statusChangeEvents || {};
  const executionMetricsEvents = aggregations.executionMetricsEvents || {};
  const messageContainingEvents = aggregations.messageContainingEvents || {};
  const executionDurationMs = executeEvents.executionDurationMs || {};
  const scheduleDelayNs = executeEvents.scheduleDelayNs || {};
  const executionsByStatus = statusChangeEvents.executionsByStatus || {};
  const gaps = executionMetricsEvents.gaps || {};
  const searchDurationMs = executionMetricsEvents.searchDurationMs || {};
  const indexingDurationMs = executionMetricsEvents.indexingDurationMs || {};
  return {
    number_of_executions: normalizeNumberOfExecutions(totalExecutions, executionsByStatus),
    number_of_logged_messages: normalizeNumberOfLoggedMessages(messageContainingEvents),
    number_of_detected_gaps: normalizeNumberOfDetectedGaps(gaps),
    schedule_delay_ms: normalizeAggregatedMetric(scheduleDelayNs, val => val / 1_000_000),
    execution_duration_ms: normalizeAggregatedMetric(executionDurationMs),
    search_duration_ms: normalizeAggregatedMetric(searchDurationMs),
    indexing_duration_ms: normalizeAggregatedMetric(indexingDurationMs),
    top_errors: aggregationLevel === 'whole-interval' ? normalizeTopErrors(messageContainingEvents) : undefined,
    top_warnings: aggregationLevel === 'whole-interval' ? normalizeTopWarnings(messageContainingEvents) : undefined
  };
};
exports.normalizeRuleExecutionStatsAggregationResult = normalizeRuleExecutionStatsAggregationResult;
const normalizeNumberOfExecutions = (totalExecutions, executionsByStatus) => {
  const getStatusCount = status => {
    const bucket = executionsByStatus.buckets.find(b => b.key === status);
    return Number((bucket === null || bucket === void 0 ? void 0 : bucket.doc_count) || 0);
  };
  return {
    total: Number(totalExecutions.value || 0),
    by_outcome: {
      succeeded: getStatusCount(_rule_monitoring.RuleExecutionStatus.succeeded),
      warning: getStatusCount(_rule_monitoring.RuleExecutionStatus['partial failure']),
      failed: getStatusCount(_rule_monitoring.RuleExecutionStatus.failed)
    }
  };
};
const normalizeNumberOfLoggedMessages = messageContainingEvents => {
  const messagesByLogLevel = messageContainingEvents.messagesByLogLevel || {};
  const getMessageCount = level => {
    const bucket = messagesByLogLevel.buckets.find(b => b.key === level);
    return Number((bucket === null || bucket === void 0 ? void 0 : bucket.doc_count) || 0);
  };
  return {
    total: Number(messageContainingEvents.doc_count || 0),
    by_level: {
      error: getMessageCount(_rule_monitoring.LogLevel.error),
      warn: getMessageCount(_rule_monitoring.LogLevel.warn),
      info: getMessageCount(_rule_monitoring.LogLevel.info),
      debug: getMessageCount(_rule_monitoring.LogLevel.debug),
      trace: getMessageCount(_rule_monitoring.LogLevel.trace)
    }
  };
};
const normalizeNumberOfDetectedGaps = gaps => {
  var _gaps$totalGapDuratio;
  return {
    total: Number(gaps.doc_count || 0),
    total_duration_s: Number(((_gaps$totalGapDuratio = gaps.totalGapDurationS) === null || _gaps$totalGapDuratio === void 0 ? void 0 : _gaps$totalGapDuratio.value) || 0)
  };
};
const normalizeAggregatedMetric = (percentilesAggregate, modifier = v => v) => {
  const rawPercentiles = percentilesAggregate.values || {};
  return {
    percentiles: (0, _lodash.mapValues)(rawPercentiles, rawValue => modifier(Number(rawValue || 0)))
  };
};
const normalizeTopErrors = messageContainingEvents => {
  var _messageContainingEve;
  const topErrors = ((_messageContainingEve = messageContainingEvents.errors) === null || _messageContainingEve === void 0 ? void 0 : _messageContainingEve.topErrors) || {};
  return normalizeTopMessages(topErrors);
};
const normalizeTopWarnings = messageContainingEvents => {
  var _messageContainingEve2;
  const topWarnings = ((_messageContainingEve2 = messageContainingEvents.warnings) === null || _messageContainingEve2 === void 0 ? void 0 : _messageContainingEve2.topWarnings) || {};
  return normalizeTopMessages(topWarnings);
};
const normalizeTopMessages = categorizeTextAggregate => {
  const buckets = (categorizeTextAggregate || {}).buckets || [];
  return buckets.map(b => {
    return {
      count: Number((b === null || b === void 0 ? void 0 : b.doc_count) || 0),
      message: String((b === null || b === void 0 ? void 0 : b.key) || '')
    };
  });
};