"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSpaceHealthRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _route_validation = require("../../../../../../utils/build_validation/route_validation");
var _utils = require("../../../../routes/utils");
var _rule_monitoring = require("../../../../../../../common/detection_engine/rule_monitoring");
var _health_timings = require("../health_timings");
var _get_space_health_request = require("./get_space_health_request");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Get health overview of the current Kibana space. Scope: all detection rules in the space.
 * Returns:
 * - health stats at the moment of the API call
 * - health stats over a specified period of time ("health interval")
 * - health stats history within the same interval in the form of a histogram
 *   (the same stats are calculated over each of the discreet sub-intervals of the whole interval)
 */
const getSpaceHealthRoute = router => {
  router.get({
    path: _rule_monitoring.GET_SPACE_HEALTH_URL,
    validate: {},
    options: {
      tags: ['access:securitySolution']
    }
  }, async (context, request, response) => {
    return handleSpaceHealthRequest({
      response,
      resolveParameters: () => (0, _get_space_health_request.validateGetSpaceHealthRequest)({}),
      resolveDependencies: async () => {
        const ctx = await context.resolve(['securitySolution']);
        const healthClient = ctx.securitySolution.getDetectionEngineHealthClient();
        return {
          healthClient
        };
      }
    });
  });
  router.post({
    path: _rule_monitoring.GET_SPACE_HEALTH_URL,
    validate: {
      body: (0, _route_validation.buildRouteValidation)(_rule_monitoring.GetSpaceHealthRequestBody)
    },
    options: {
      tags: ['access:securitySolution']
    }
  }, async (context, request, response) => {
    return handleSpaceHealthRequest({
      response,
      resolveParameters: () => (0, _get_space_health_request.validateGetSpaceHealthRequest)(request.body),
      resolveDependencies: async () => {
        const ctx = await context.resolve(['securitySolution']);
        const healthClient = ctx.securitySolution.getDetectionEngineHealthClient();
        return {
          healthClient
        };
      }
    });
  });
};
exports.getSpaceHealthRoute = getSpaceHealthRoute;
const handleSpaceHealthRequest = async args => {
  const {
    response,
    resolveParameters,
    resolveDependencies
  } = args;
  const siemResponse = (0, _utils.buildSiemResponse)(response);
  try {
    const params = resolveParameters();
    const {
      healthClient
    } = await resolveDependencies();
    const spaceHealthParameters = {
      interval: params.interval
    };
    const spaceHealth = await healthClient.calculateSpaceHealth(spaceHealthParameters);
    const responseBody = {
      timings: (0, _health_timings.calculateHealthTimings)(params.requestReceivedAt),
      parameters: spaceHealthParameters,
      health: {
        ...spaceHealth,
        debug: params.debug ? spaceHealth.debug : undefined
      }
    };
    return response.ok({
      body: responseBody
    });
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    return siemResponse.error({
      body: error.message,
      statusCode: error.statusCode
    });
  }
};