"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performRuleUpgradeRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _prebuilt_rules = require("../../../../../../common/detection_engine/prebuilt_rules");
var _perform_rule_upgrade_request_schema = require("../../../../../../common/detection_engine/prebuilt_rules/api/perform_rule_upgrade/perform_rule_upgrade_request_schema");
var _perform_rule_upgrade_response_schema = require("../../../../../../common/detection_engine/prebuilt_rules/api/perform_rule_upgrade/perform_rule_upgrade_response_schema");
var _utility_types = require("../../../../../../common/utility_types");
var _route_validation = require("../../../../../utils/build_validation/route_validation");
var _utils = require("../../../routes/utils");
var _rule_converters = require("../../../rule_management/normalization/rule_converters");
var _aggregate_prebuilt_rule_errors = require("../../logic/aggregate_prebuilt_rule_errors");
var _perform_timelines_installation = require("../../logic/perform_timelines_installation");
var _prebuilt_rule_assets_client = require("../../logic/rule_assets/prebuilt_rule_assets_client");
var _prebuilt_rule_objects_client = require("../../logic/rule_objects/prebuilt_rule_objects_client");
var _upgrade_prebuilt_rules = require("../../logic/rule_objects/upgrade_prebuilt_rules");
var _fetch_rule_versions_triad = require("../../logic/rule_versions/fetch_rule_versions_triad");
var _get_version_buckets = require("../../model/rule_versions/get_version_buckets");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const performRuleUpgradeRoute = router => {
  router.post({
    path: _prebuilt_rules.PERFORM_RULE_UPGRADE_URL,
    validate: {
      body: (0, _route_validation.buildRouteValidation)(_perform_rule_upgrade_request_schema.PerformRuleUpgradeRequestBody)
    },
    options: {
      tags: ['access:securitySolution']
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const ctx = await context.resolve(['core', 'alerting', 'securitySolution']);
      const soClient = ctx.core.savedObjects.client;
      const rulesClient = ctx.alerting.getRulesClient();
      const ruleAssetsClient = (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(soClient);
      const ruleObjectsClient = (0, _prebuilt_rule_objects_client.createPrebuiltRuleObjectsClient)(rulesClient);
      const {
        mode,
        pick_version: globalPickVersion = _perform_rule_upgrade_request_schema.PickVersionValues.TARGET
      } = request.body;
      const fetchErrors = [];
      const targetRules = [];
      const skippedRules = [];
      const versionSpecifiers = mode === 'ALL_RULES' ? undefined : request.body.rules;
      const versionSpecifiersMap = new Map(versionSpecifiers === null || versionSpecifiers === void 0 ? void 0 : versionSpecifiers.map(rule => [rule.rule_id, rule]));
      const ruleVersionsMap = await (0, _fetch_rule_versions_triad.fetchRuleVersionsTriad)({
        ruleAssetsClient,
        ruleObjectsClient,
        versionSpecifiers
      });
      const versionBuckets = (0, _get_version_buckets.getVersionBuckets)(ruleVersionsMap);
      const {
        currentRules
      } = versionBuckets;
      // The upgradeable rules list is mutable; we can remove rules from it because of version mismatch
      let upgradeableRules = versionBuckets.upgradeableRules;

      // Perform all the checks we can before we start the upgrade process
      if (mode === 'SPECIFIC_RULES') {
        const installedRuleIds = new Set(currentRules.map(rule => rule.rule_id));
        const upgradeableRuleIds = new Set(upgradeableRules.map(({
          current
        }) => current.rule_id));
        request.body.rules.forEach(rule => {
          var _ruleVersionsMap$get, _ruleVersionsMap$get$;
          // Check that the requested rule was found
          if (!installedRuleIds.has(rule.rule_id)) {
            fetchErrors.push({
              error: new Error(`Rule with ID "${rule.rule_id}" and version "${rule.version}" not found`),
              item: rule
            });
            return;
          }

          // Check that the requested rule is upgradeable
          if (!upgradeableRuleIds.has(rule.rule_id)) {
            skippedRules.push({
              rule_id: rule.rule_id,
              reason: _perform_rule_upgrade_response_schema.SkipRuleUpgradeReason.RULE_UP_TO_DATE
            });
            return;
          }

          // Check that rule revisions match (no update slipped in since the user reviewed the list)
          const currentRevision = (_ruleVersionsMap$get = ruleVersionsMap.get(rule.rule_id)) === null || _ruleVersionsMap$get === void 0 ? void 0 : (_ruleVersionsMap$get$ = _ruleVersionsMap$get.current) === null || _ruleVersionsMap$get$ === void 0 ? void 0 : _ruleVersionsMap$get$.revision;
          if (rule.revision !== currentRevision) {
            fetchErrors.push({
              error: new Error(`Revision mismatch for rule ID ${rule.rule_id}: expected ${rule.revision}, got ${currentRevision}`),
              item: rule
            });
            // Remove the rule from the list of upgradeable rules
            upgradeableRules = upgradeableRules.filter(({
              current
            }) => current.rule_id !== rule.rule_id);
          }
        });
      }

      // Construct the list of target rule versions
      upgradeableRules.forEach(({
        current,
        target
      }) => {
        var _versionSpecifiersMap, _versionSpecifiersMap2, _ruleVersionsMap$get2;
        const rulePickVersion = (_versionSpecifiersMap = versionSpecifiersMap === null || versionSpecifiersMap === void 0 ? void 0 : (_versionSpecifiersMap2 = versionSpecifiersMap.get(current.rule_id)) === null || _versionSpecifiersMap2 === void 0 ? void 0 : _versionSpecifiersMap2.pick_version) !== null && _versionSpecifiersMap !== void 0 ? _versionSpecifiersMap : globalPickVersion;
        switch (rulePickVersion) {
          case _perform_rule_upgrade_request_schema.PickVersionValues.BASE:
            const baseVersion = (_ruleVersionsMap$get2 = ruleVersionsMap.get(current.rule_id)) === null || _ruleVersionsMap$get2 === void 0 ? void 0 : _ruleVersionsMap$get2.base;
            if (baseVersion) {
              targetRules.push({
                ...baseVersion,
                version: target.version
              });
            } else {
              fetchErrors.push({
                error: new Error(`Could not find base version for rule ${current.rule_id}`),
                item: current
              });
            }
            break;
          case _perform_rule_upgrade_request_schema.PickVersionValues.CURRENT:
            targetRules.push({
              ...current,
              version: target.version
            });
            break;
          case _perform_rule_upgrade_request_schema.PickVersionValues.TARGET:
            targetRules.push(target);
            break;
          default:
            (0, _utility_types.assertUnreachable)(rulePickVersion);
        }
      });

      // Perform the upgrade
      const {
        results: updatedRules,
        errors: installationErrors
      } = await (0, _upgrade_prebuilt_rules.upgradePrebuiltRules)(rulesClient, targetRules);
      const ruleErrors = [...fetchErrors, ...installationErrors];
      const {
        error: timelineInstallationError
      } = await (0, _perform_timelines_installation.performTimelinesInstallation)(ctx.securitySolution);
      const allErrors = (0, _aggregate_prebuilt_rule_errors.aggregatePrebuiltRuleErrors)(ruleErrors);
      if (timelineInstallationError) {
        allErrors.push({
          message: timelineInstallationError,
          rules: []
        });
      }
      const body = {
        summary: {
          total: updatedRules.length + skippedRules.length + ruleErrors.length,
          skipped: skippedRules.length,
          succeeded: updatedRules.length,
          failed: ruleErrors.length
        },
        results: {
          updated: updatedRules.map(({
            result
          }) => (0, _rule_converters.internalRuleToAPIResponse)(result)),
          skipped: skippedRules
        },
        errors: allErrors
      };
      return response.ok({
        body
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.performRuleUpgradeRoute = performRuleUpgradeRoute;