"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AppFeaturesConfigMerger = void 0;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AppFeaturesConfigMerger {
  constructor(logger, subFeaturesMap) {
    this.logger = logger;
    this.subFeaturesMap = subFeaturesMap;
  }

  /**
   * Merges `appFeaturesConfigs` into `kibanaFeatureConfig`.
   * @param kibanaFeatureConfig the KibanaFeatureConfig to merge into
   * @param appFeaturesConfigs the AppFeatureKibanaConfig to merge
   * @returns mergedKibanaFeatureConfig the merged KibanaFeatureConfig
   * */
  mergeAppFeatureConfigs(kibanaFeatureConfig, kibanaSubFeatureIds, appFeaturesConfigs) {
    const mergedKibanaFeatureConfig = (0, _lodash.cloneDeep)(kibanaFeatureConfig);
    const subFeaturesPrivilegesToMerge = [];
    const enabledSubFeaturesIndexed = Object.fromEntries(kibanaSubFeatureIds.map(id => [id, true]));
    appFeaturesConfigs.forEach(appFeatureConfig => {
      const {
        subFeaturesPrivileges,
        subFeatureIds,
        ...appFeatureConfigToMerge
      } = (0, _lodash.cloneDeep)(appFeatureConfig);
      subFeatureIds === null || subFeatureIds === void 0 ? void 0 : subFeatureIds.forEach(subFeatureId => {
        enabledSubFeaturesIndexed[subFeatureId] = true;
      });
      if (subFeaturesPrivileges) {
        subFeaturesPrivilegesToMerge.push(...subFeaturesPrivileges);
      }
      (0, _lodash.mergeWith)(mergedKibanaFeatureConfig, appFeatureConfigToMerge, featureConfigMerger);
    });

    // generate sub features configs from enabled sub feature ids, preserving map order
    const mergedKibanaSubFeatures = [];
    this.subFeaturesMap.forEach((subFeature, id) => {
      if (enabledSubFeaturesIndexed[id]) {
        mergedKibanaSubFeatures.push((0, _lodash.cloneDeep)(subFeature));
      }
    });

    // add extra privileges to subFeatures
    subFeaturesPrivilegesToMerge.forEach(subFeaturesPrivileges => {
      this.mergeSubFeaturesPrivileges(mergedKibanaSubFeatures, subFeaturesPrivileges);
    });
    mergedKibanaFeatureConfig.subFeatures = mergedKibanaSubFeatures;
    return mergedKibanaFeatureConfig;
  }

  /**
   * Merges `subFeaturesPrivileges` into `kibanaFeatureConfig.subFeatures` by finding the subFeature privilege id.
   * @param subFeatures the subFeatures to merge into
   * @param subFeaturesPrivileges the subFeaturesPrivileges to merge
   * @returns void
   * */
  mergeSubFeaturesPrivileges(subFeatures, subFeaturesPrivileges) {
    const merged = subFeatures.find(({
      privilegeGroups
    }) => privilegeGroups.some(({
      privileges
    }) => {
      const subFeaturePrivilegeToUpdate = privileges.find(({
        id
      }) => id === subFeaturesPrivileges.id);
      if (subFeaturePrivilegeToUpdate) {
        (0, _lodash.mergeWith)(subFeaturePrivilegeToUpdate, subFeaturesPrivileges, featureConfigMerger);
        return true;
      }
      return false;
    }));
    if (!merged) {
      this.logger.warn(`Trying to merge subFeaturesPrivileges ${subFeaturesPrivileges.id} but the subFeature privilege was not found`);
    }
  }
}

/**
 * The customizer used by lodash.mergeWith to merge deep objects
 * Uses concatenation for arrays and removes duplicates, objects are merged using lodash.mergeWith default behavior
 * */
exports.AppFeaturesConfigMerger = AppFeaturesConfigMerger;
function featureConfigMerger(objValue, srcValue) {
  if ((0, _lodash.isArray)(srcValue)) {
    if ((0, _lodash.isArray)(objValue)) {
      return (0, _lodash.uniq)(objValue.concat(srcValue));
    }
    return srcValue;
  }
}