"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.endpointMiddlewareFactory = void 0;
exports.handleLoadMetadataTransformStats = handleLoadMetadataTransformStats;
var _rxjs = require("rxjs");
var _gte = _interopRequireDefault(require("semver/functions/gte"));
var _constants = require("../../../../../common/endpoint/constants");
var _endpoint_isolation = require("../../../../common/lib/endpoint_isolation");
var _endpoint_pending_actions = require("../../../../common/lib/endpoint_pending_actions");
var _resolve_path_variables = require("../../../../common/utils/resolve_path_variables");
var _policies = require("../../../services/policies/policies");
var _state = require("../../../state");
var _ingest = require("../../../services/policies/ingest");
var _selectors = require("./selectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// eslint-disable-next-line no-console
const logError = console.error;
const endpointMiddlewareFactory = (coreStart, depsStart) => {
  // this needs to be called after endpointPackageVersion is loaded (getEndpointPackageInfo)
  // or else wrong pattern might be loaded
  async function fetchIndexPatterns(state) {
    var _endpointPackageVersi;
    const packageVersion = (_endpointPackageVersi = (0, _selectors.endpointPackageVersion)(state)) !== null && _endpointPackageVersi !== void 0 ? _endpointPackageVersi : '';
    const parsedPackageVersion = packageVersion.includes('-') ? packageVersion.substring(0, packageVersion.indexOf('-')) : packageVersion;
    const minUnitedIndexVersion = '1.2.0';
    const indexPatternToFetch = (0, _gte.default)(parsedPackageVersion, minUnitedIndexVersion) ? _constants.METADATA_UNITED_INDEX : _constants.metadataCurrentIndexPattern;
    const res$ = depsStart.data.search.search({
      indices: [indexPatternToFetch],
      onlyCheckIfIndicesExist: false
    }, {
      strategy: _constants.ENDPOINT_FIELDS_SEARCH_STRATEGY
    });
    const response = await (0, _rxjs.firstValueFrom)(res$);
    const indexPattern = {
      title: indexPatternToFetch,
      fields: response.indexFields
    };
    return [indexPattern];
  }
  return store => next => async action => {
    next(action);
    const {
      getState,
      dispatch
    } = store;
    await getEndpointPackageInfo(getState(), dispatch, coreStart);

    // Endpoint list
    if ((action.type === 'userChangedUrl' || action.type === 'appRequestedEndpointList') && (0, _selectors.isOnEndpointPage)(getState()) && !(0, _selectors.hasSelectedEndpoint)(getState())) {
      await endpointDetailsListMiddleware({
        coreStart,
        store,
        fetchIndexPatterns
      });
    }

    // Endpoint Details
    if (action.type === 'userChangedUrl' && (0, _selectors.hasSelectedEndpoint)(getState())) {
      const {
        selected_endpoint: selectedEndpoint
      } = (0, _selectors.uiQueryParams)(getState());
      await endpointDetailsMiddleware({
        store,
        coreStart,
        selectedEndpoint
      });
    }
    if (action.type === 'endpointDetailsLoad') {
      await loadEndpointDetails({
        store,
        coreStart,
        selectedEndpoint: action.payload.endpointId
      });
    }

    // Isolate Host
    if (action.type === 'endpointIsolationRequest') {
      return handleIsolateEndpointHost(store, action);
    }
    if (action.type === 'loadMetadataTransformStats') {
      return handleLoadMetadataTransformStats(coreStart.http, store);
    }
  };
};
exports.endpointMiddlewareFactory = endpointMiddlewareFactory;
const getAgentAndPoliciesForEndpointsList = async (http, hosts, currentNonExistingPolicies) => {
  if (hosts.length === 0) {
    return;
  }

  // Create an array of unique policy IDs that are not yet known to be non-existing.
  const policyIdsToCheck = [...new Set(hosts.reduce((acc, host) => {
    const appliedPolicyId = host.metadata.Endpoint.policy.applied.id;
    if (!currentNonExistingPolicies[appliedPolicyId]) {
      acc.push(appliedPolicyId);
    }
    return acc;
  }, []))];
  if (policyIdsToCheck.length === 0) {
    return;
  }

  // We use the Agent Policy API here, instead of the Package Policy, because we can't use
  // filter by ID of the Saved Object. Agent Policy, however, keeps a reference (array) of
  // Package Ids that it uses, thus if a reference exists there, then the package policy (policy)
  // exists.
  const policiesFound = (await (0, _ingest.sendBulkGetPackagePolicies)(http, policyIdsToCheck)).items.reduce((list, packagePolicy) => {
    list.packagePolicy[packagePolicy.id] = true;
    list.agentPolicy[packagePolicy.id] = packagePolicy.policy_id;
    return list;
  }, {
    packagePolicy: {},
    agentPolicy: {}
  });

  // packagePolicy contains non-existing packagePolicy ids whereas agentPolicy contains existing agentPolicy ids
  const nonExistingPackagePoliciesAndExistingAgentPolicies = policyIdsToCheck.reduce((list, policyId) => {
    if (policiesFound.packagePolicy[policyId]) {
      list.agentPolicy[policyId] = policiesFound.agentPolicy[policyId];
      return list;
    }
    list.packagePolicy[policyId] = true;
    return list;
  }, {
    packagePolicy: {},
    agentPolicy: {}
  });
  if (Object.keys(nonExistingPackagePoliciesAndExistingAgentPolicies.packagePolicy).length === 0 && Object.keys(nonExistingPackagePoliciesAndExistingAgentPolicies.agentPolicy).length === 0) {
    return;
  }
  return nonExistingPackagePoliciesAndExistingAgentPolicies;
};
const endpointsTotal = async http => {
  try {
    return (await http.get(_constants.HOST_METADATA_LIST_ROUTE, {
      query: {
        page: 0,
        pageSize: 1
      }
    })).total;
  } catch (error) {
    // TODO should handle the error instead of logging it to the browser
    // Also this is an anti-pattern we shouldn't use
    logError(`error while trying to check for total endpoints`);
    logError(error);
  }
  return 0;
};
const doEndpointsExist = async http => {
  try {
    return (await endpointsTotal(http)) > 0;
  } catch (error) {
    // TODO should handle the error instead of logging it to the browser
    // Also this is an anti-pattern we shouldn't use
    logError(`error while trying to check if endpoints exist`);
    logError(error);
  }
  return false;
};
const handleIsolateEndpointHost = async ({
  getState,
  dispatch
}, action) => {
  const state = getState();
  if ((0, _selectors.getIsIsolationRequestPending)(state)) {
    return;
  }
  dispatch({
    type: 'endpointIsolationRequestStateChange',
    payload: (0, _state.createLoadingResourceState)((0, _state.asStaleResourceState)((0, _selectors.getCurrentIsolationRequestState)(state)))
  });
  try {
    // Cast needed below due to the value of payload being `Immutable<>`
    let response;
    if (action.payload.type === 'unisolate') {
      response = await (0, _endpoint_isolation.unIsolateHost)(action.payload.data);
    } else {
      response = await (0, _endpoint_isolation.isolateHost)(action.payload.data);
    }
    dispatch({
      type: 'endpointIsolationRequestStateChange',
      payload: (0, _state.createLoadedResourceState)(response)
    });
  } catch (error) {
    var _error$body;
    dispatch({
      type: 'endpointIsolationRequestStateChange',
      payload: (0, _state.createFailedResourceState)((_error$body = error.body) !== null && _error$body !== void 0 ? _error$body : error)
    });
  }
};
async function getEndpointPackageInfo(state, dispatch, coreStart) {
  if (!(0, _selectors.getIsEndpointPackageInfoUninitialized)(state)) return;
  dispatch({
    type: 'endpointPackageInfoStateChanged',
    payload: (0, _state.createLoadingResourceState)((0, _state.asStaleResourceState)((0, _selectors.endpointPackageInfo)(state)))
  });
  try {
    const packageInfo = await (0, _ingest.sendGetEndpointSecurityPackage)(coreStart.http);
    dispatch({
      type: 'endpointPackageInfoStateChanged',
      payload: (0, _state.createLoadedResourceState)(packageInfo)
    });
  } catch (error) {
    // TODO should handle the error instead of logging it to the browser
    // Also this is an anti-pattern we shouldn't use
    // Ignore Errors, since this should not hinder the user's ability to use the UI
    logError(error);
    dispatch({
      type: 'endpointPackageInfoStateChanged',
      payload: (0, _state.createFailedResourceState)(error)
    });
  }
}

/**
 * retrieves the Endpoint pending actions for all of the existing endpoints being displayed on the list
 * or the details tab.
 *
 * @param store
 */
const loadEndpointsPendingActions = async ({
  getState,
  dispatch
}) => {
  const state = getState();
  const detailsEndpoint = (0, _selectors.detailsData)(state);
  const listEndpoints = (0, _selectors.listData)(state);
  const agentsIds = new Set();

  // get all agent ids for the endpoints in the list
  if (detailsEndpoint) {
    agentsIds.add(detailsEndpoint.elastic.agent.id);
  }
  for (const endpointInfo of listEndpoints) {
    agentsIds.add(endpointInfo.metadata.elastic.agent.id);
  }
  if (agentsIds.size === 0) {
    return;
  }
  try {
    const {
      data: pendingActions
    } = await (0, _endpoint_pending_actions.fetchPendingActionsByAgentId)(Array.from(agentsIds));
    const agentIdToPendingActions = new Map();
    for (const pendingAction of pendingActions) {
      agentIdToPendingActions.set(pendingAction.agent_id, pendingAction.pending_actions);
    }
    dispatch({
      type: 'endpointPendingActionsStateChanged',
      payload: (0, _state.createLoadedResourceState)(agentIdToPendingActions)
    });
  } catch (error) {
    // TODO should handle the error instead of logging it to the browser
    // Also this is an anti-pattern we shouldn't use
    logError(error);
  }
};
async function endpointDetailsListMiddleware({
  store,
  coreStart,
  fetchIndexPatterns
}) {
  const {
    getState,
    dispatch
  } = store;
  const {
    page_index: pageIndex,
    page_size: pageSize
  } = (0, _selectors.uiQueryParams)(getState());
  let endpointResponse;
  try {
    const decodedQuery = (0, _selectors.searchBarQuery)(getState());
    endpointResponse = await coreStart.http.get(_constants.HOST_METADATA_LIST_ROUTE, {
      query: {
        page: pageIndex,
        pageSize,
        kuery: decodedQuery.query
      }
    });
    dispatch({
      type: 'serverReturnedEndpointList',
      payload: endpointResponse
    });
    loadEndpointsPendingActions(store);
    dispatchIngestPolicies({
      http: coreStart.http,
      hosts: endpointResponse.data,
      store
    });
  } catch (error) {
    dispatch({
      type: 'serverFailedToReturnEndpointList',
      payload: error
    });
  }

  // get index pattern and fields for search bar
  if ((0, _selectors.patterns)(getState()).length === 0) {
    try {
      const indexPatterns = await fetchIndexPatterns(getState());
      if (indexPatterns !== undefined) {
        dispatch({
          type: 'serverReturnedMetadataPatterns',
          payload: indexPatterns
        });
      }
    } catch (error) {
      dispatch({
        type: 'serverFailedToReturnMetadataPatterns',
        payload: error
      });
    }
  }

  // No endpoints, so we should check to see if there are policies for onboarding
  if (endpointResponse && endpointResponse.data.length === 0) {
    const http = coreStart.http;

    // The original query to the list could have had an invalid param (ex. invalid page_size),
    // so we check first if endpoints actually do exist before pulling in data for the onboarding
    // messages.
    if (await doEndpointsExist(http)) {
      return;
    }
    dispatch({
      type: 'serverReturnedEndpointExistValue',
      payload: false
    });
    try {
      const policyDataResponse = await (0, _policies.sendGetEndpointSpecificPackagePolicies)(http, {
        query: {
          perPage: 50,
          // Since this is an oboarding flow, we'll cap at 50 policies.
          page: 1
        }
      });
      dispatch({
        type: 'serverReturnedPoliciesForOnboarding',
        payload: {
          policyItems: policyDataResponse.items
        }
      });
    } catch (error) {
      var _error$body2;
      dispatch({
        type: 'serverFailedToReturnPoliciesForOnboarding',
        payload: (_error$body2 = error.body) !== null && _error$body2 !== void 0 ? _error$body2 : error
      });
    }
  } else {
    dispatch({
      type: 'serverCancelledPolicyItemsLoading'
    });
    dispatch({
      type: 'serverReturnedEndpointExistValue',
      payload: true
    });
  }
}
async function loadEndpointDetails({
  selectedEndpoint,
  store,
  coreStart
}) {
  const {
    getState,
    dispatch
  } = store;
  // call the endpoint details api
  try {
    const response = await coreStart.http.get((0, _resolve_path_variables.resolvePathVariables)(_constants.HOST_METADATA_GET_ROUTE, {
      id: selectedEndpoint
    }));
    dispatch({
      type: 'serverReturnedEndpointDetails',
      payload: response
    });
    try {
      const ingestPolicies = await getAgentAndPoliciesForEndpointsList(coreStart.http, [response], (0, _selectors.nonExistingPolicies)(getState()));
      if (ingestPolicies !== undefined) {
        dispatch({
          type: 'serverReturnedEndpointNonExistingPolicies',
          payload: ingestPolicies.packagePolicy
        });
      }
      if ((ingestPolicies === null || ingestPolicies === void 0 ? void 0 : ingestPolicies.agentPolicy) !== undefined) {
        dispatch({
          type: 'serverReturnedEndpointAgentPolicies',
          payload: ingestPolicies.agentPolicy
        });
      }
    } catch (error) {
      // TODO should handle the error instead of logging it to the browser
      // Also this is an anti-pattern we shouldn't use
      // Ignore Errors, since this should not hinder the user's ability to use the UI
      logError(error);
    }
  } catch (error) {
    dispatch({
      type: 'serverFailedToReturnEndpointDetails',
      payload: error
    });
  }
  loadEndpointsPendingActions(store);

  // call the policy response api
  try {
    const policyResponse = await coreStart.http.get(_constants.BASE_POLICY_RESPONSE_ROUTE, {
      query: {
        agentId: selectedEndpoint
      }
    });
    dispatch({
      type: 'serverReturnedEndpointPolicyResponse',
      payload: policyResponse
    });
  } catch (error) {
    dispatch({
      type: 'serverFailedToReturnEndpointPolicyResponse',
      payload: error
    });
  }
}
async function endpointDetailsMiddleware({
  coreStart,
  selectedEndpoint,
  store
}) {
  const {
    getState,
    dispatch
  } = store;
  dispatch({
    type: 'serverCancelledPolicyItemsLoading'
  });

  // If user navigated directly to a endpoint details page, load the endpoint list
  if ((0, _selectors.listData)(getState()).length === 0) {
    const {
      page_index: pageIndex,
      page_size: pageSize
    } = (0, _selectors.uiQueryParams)(getState());
    try {
      const response = await coreStart.http.get(_constants.HOST_METADATA_LIST_ROUTE, {
        query: {
          page: pageIndex,
          pageSize
        }
      });
      dispatch({
        type: 'serverReturnedEndpointList',
        payload: response
      });
      dispatchIngestPolicies({
        http: coreStart.http,
        hosts: response.data,
        store
      });
    } catch (error) {
      dispatch({
        type: 'serverFailedToReturnEndpointList',
        payload: error
      });
    }
  } else {
    dispatch({
      type: 'serverCancelledEndpointListLoading'
    });
  }
  if (typeof selectedEndpoint === 'undefined') {
    return;
  }
  await loadEndpointDetails({
    store,
    coreStart,
    selectedEndpoint
  });
}
async function handleLoadMetadataTransformStats(http, store) {
  const {
    getState,
    dispatch
  } = store;
  if (!http || !getState || !dispatch) {
    return;
  }
  const state = getState();
  if ((0, _selectors.isMetadataTransformStatsLoading)(state)) return;
  dispatch({
    type: 'metadataTransformStatsChanged',
    payload: (0, _state.createLoadingResourceState)((0, _state.asStaleResourceState)((0, _selectors.getMetadataTransformStats)(state)))
  });
  try {
    const transformStatsResponse = await http.get(_constants.METADATA_TRANSFORMS_STATUS_ROUTE);
    dispatch({
      type: 'metadataTransformStatsChanged',
      payload: (0, _state.createLoadedResourceState)(transformStatsResponse.transforms)
    });
  } catch (error) {
    dispatch({
      type: 'metadataTransformStatsChanged',
      payload: (0, _state.createFailedResourceState)(error)
    });
  }
}
async function dispatchIngestPolicies({
  store,
  hosts,
  http
}) {
  const {
    getState,
    dispatch
  } = store;
  try {
    const ingestPolicies = await getAgentAndPoliciesForEndpointsList(http, hosts, (0, _selectors.nonExistingPolicies)(getState()));
    if ((ingestPolicies === null || ingestPolicies === void 0 ? void 0 : ingestPolicies.packagePolicy) !== undefined) {
      dispatch({
        type: 'serverReturnedEndpointNonExistingPolicies',
        payload: ingestPolicies.packagePolicy
      });
    }
    if ((ingestPolicies === null || ingestPolicies === void 0 ? void 0 : ingestPolicies.agentPolicy) !== undefined) {
      dispatch({
        type: 'serverReturnedEndpointAgentPolicies',
        payload: ingestPolicies.agentPolicy
      });
    }
  } catch (error) {
    // TODO should handle the error instead of logging it to the browser
    // Also this is an anti-pattern we shouldn't use
    // Ignore Errors, since this should not hinder the user's ability to use the UI
    logError(error);
  }
}