"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFetchFieldValuePairWithAggregation = void 0;
var _esQuery = require("@kbn/es-query");
var _reactQuery = require("@tanstack/react-query");
var _build_requests = require("../utils/build_requests");
var _fetch_data = require("../utils/fetch_data");
var _kibana = require("../../../common/lib/kibana");
var _store = require("../../../common/store");
var _use_selector = require("../../../common/hooks/use_selector");
var _use_global_time = require("../../../common/containers/use_global_time");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const QUERY_KEY = 'useFetchFieldValuePairWithAggregation';
/**
 * Hook to retrieve all the unique documents for the aggregationField in the environment that have the field/value pair, using ReactQuery.
 *
 * Foe example, passing 'host.name' via the aggregationField props will return the number of unique hosts in the environment that have the field/value pair.
 */
const useFetchFieldValuePairWithAggregation = ({
  highlightedField,
  isActiveTimelines,
  aggregationField
}) => {
  const {
    services: {
      data: {
        search: searchService
      }
    }
  } = (0, _kibana.useKibana)();
  const timelineTime = (0, _use_selector.useDeepEqualSelector)(state => _store.inputsSelectors.timelineTimeRangeSelector(state));
  const globalTime = (0, _use_global_time.useGlobalTime)();
  const {
    to,
    from
  } = isActiveTimelines ? timelineTime : globalTime;
  const {
    name,
    values
  } = highlightedField;
  const searchRequest = buildSearchRequest(name, values, from, to, aggregationField);
  const {
    data,
    isLoading,
    isError
  } = (0, _reactQuery.useQuery)([QUERY_KEY, name, values, from, to, aggregationField], () => (0, _fetch_data.createFetchData)(searchService, searchRequest), {
    select: res => res.aggregations[_fetch_data.AGG_KEY].buckets.length,
    keepPreviousData: true
  });
  return {
    loading: isLoading,
    error: isError,
    count: data || 0
  };
};

/**
 * Build the search request for the field/values pair, for a date range from/to.
 * The request contains aggregation by aggregationField.
 */
exports.useFetchFieldValuePairWithAggregation = useFetchFieldValuePairWithAggregation;
const buildSearchRequest = (field, values, from, to, aggregationField) => {
  const query = (0, _esQuery.buildEsQuery)(undefined, [], [{
    query: {
      bool: {
        filter: [{
          match: {
            [field]: values[0]
          }
        }, {
          range: {
            '@timestamp': {
              gte: from,
              lte: to
            }
          }
        }]
      }
    },
    meta: {}
  }]);
  return (0, _build_requests.buildAggregationSearchRequest)(aggregationField, _fetch_data.AGG_KEY, query);
};