"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EndpointMetadataGenerator = void 0;
var _lodash = require("lodash");
var _semver = require("semver");
var _base_data_generator = require("./base_data_generator");
var _types = require("../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable max-classes-per-file */

/**
 * Metadata generator for docs that are sent by the Endpoint running on hosts
 */
class EndpointMetadataGenerator extends _base_data_generator.BaseDataGenerator {
  /**
   * Returns a Custom `EndpointMetadataGenerator` subclass that will generate specific
   * documents based on input arguments
   */
  static custom({
    version,
    os,
    isolation
  } = {}) {
    return class extends EndpointMetadataGenerator {
      generate(overrides = {}) {
        if (version) {
          (0, _lodash.set)(overrides, 'agent.version', version);
        }
        if (os) {
          switch (os) {
            case 'linux':
              (0, _lodash.set)(overrides, 'host.os', EndpointMetadataGenerator.linuxOSFields);
              break;
            case 'macOS':
              (0, _lodash.set)(overrides, 'host.os', EndpointMetadataGenerator.macOSFields);
              break;
            default:
              (0, _lodash.set)(overrides, 'host.os', EndpointMetadataGenerator.windowsOSFields);
          }
        }
        if (isolation !== undefined) {
          (0, _lodash.set)(overrides, 'Endpoint.state.isolation', isolation);
        }
        return super.generate(overrides);
      }
    };
  }
  static get windowsOSFields() {
    return {
      name: 'Windows',
      full: 'Windows 10',
      version: '10.0',
      platform: 'Windows',
      family: 'windows',
      Ext: {
        variant: 'Windows Pro'
      }
    };
  }
  static get linuxOSFields() {
    return {
      Ext: {
        variant: 'Debian'
      },
      kernel: '4.19.0-21-cloud-amd64 #1 SMP Debian 4.19.249-2 (2022-06-30)',
      name: 'Linux',
      family: 'debian',
      type: 'linux',
      version: '10.12',
      platform: 'debian',
      full: 'Debian 10.12'
    };
  }
  static get macOSFields() {
    return {
      name: 'macOS',
      full: 'macOS Monterey',
      version: '12.6.1',
      platform: 'macOS',
      family: 'Darwin',
      Ext: {
        variant: 'Darwin'
      }
    };
  }

  /** Generate an Endpoint host metadata document */
  generate(overrides = {}) {
    var _overrides$Timestamp, _overrides$host$hostn, _overrides$host, _overrides$agent$vers, _overrides$agent, _overrides$Endpoint$s, _overrides$Endpoint, _overrides$Endpoint$s2;
    const ts = (_overrides$Timestamp = overrides['@timestamp']) !== null && _overrides$Timestamp !== void 0 ? _overrides$Timestamp : new Date().getTime();
    const hostName = (_overrides$host$hostn = overrides === null || overrides === void 0 ? void 0 : (_overrides$host = overrides.host) === null || _overrides$host === void 0 ? void 0 : _overrides$host.hostname) !== null && _overrides$host$hostn !== void 0 ? _overrides$host$hostn : this.randomHostname();
    const agentVersion = (_overrides$agent$vers = overrides === null || overrides === void 0 ? void 0 : (_overrides$agent = overrides.agent) === null || _overrides$agent === void 0 ? void 0 : _overrides$agent.version) !== null && _overrides$agent$vers !== void 0 ? _overrides$agent$vers : this.randomVersion();
    const agentId = this.seededUUIDv4();
    const isIsolated = (_overrides$Endpoint$s = overrides === null || overrides === void 0 ? void 0 : (_overrides$Endpoint = overrides.Endpoint) === null || _overrides$Endpoint === void 0 ? void 0 : (_overrides$Endpoint$s2 = _overrides$Endpoint.state) === null || _overrides$Endpoint$s2 === void 0 ? void 0 : _overrides$Endpoint$s2.isolation) !== null && _overrides$Endpoint$s !== void 0 ? _overrides$Endpoint$s : this.randomBoolean(0.3);
    const capabilities = ['isolation'];

    // v8.4 introduced additional endpoint capabilities
    if ((0, _semver.gte)(agentVersion, '8.4.0')) {
      capabilities.push('kill_process', 'suspend_process', 'running_processes');
    }
    if ((0, _semver.gte)(agentVersion, '8.6.0')) {
      capabilities.push('get_file');
    }

    // v8.8 introduced execute capability
    if ((0, _semver.gte)(agentVersion, '8.8.0')) {
      capabilities.push('execute');
    }

    // v8.9 introduced `upload` capability
    if ((0, _semver.gte)(agentVersion, '8.9.0')) {
      capabilities.push('upload_file');
    }
    const hostMetadataDoc = {
      '@timestamp': ts,
      event: {
        created: ts,
        id: this.seededUUIDv4(),
        kind: 'metric',
        category: ['host'],
        type: ['info'],
        module: 'endpoint',
        action: 'endpoint_metadata',
        dataset: 'endpoint.metadata'
      },
      data_stream: {
        type: 'metrics',
        dataset: 'endpoint.metadata',
        namespace: 'default'
      },
      agent: {
        version: agentVersion,
        id: agentId,
        type: 'endpoint'
      },
      elastic: {
        agent: {
          id: agentId
        }
      },
      host: {
        id: this.seededUUIDv4(),
        hostname: hostName,
        name: hostName,
        architecture: this.randomString(10),
        ip: this.randomArray(3, () => this.randomIP()),
        mac: this.randomArray(3, () => this.randomMac()),
        os: this.randomOsFields()
      },
      Endpoint: {
        status: _types.EndpointStatus.enrolled,
        policy: {
          applied: {
            name: 'With Eventing',
            id: 'C2A9093E-E289-4C0A-AA44-8C32A414FA7A',
            status: _types.HostPolicyResponseActionStatus.success,
            endpoint_policy_version: 3,
            version: 5
          }
        },
        configuration: {
          isolation: isIsolated
        },
        state: {
          isolation: isIsolated
        },
        capabilities
      }
    };
    return (0, _lodash.merge)(hostMetadataDoc, overrides);
  }

  /** Generates the complete `HostInfo` as returned by a call to the Endpoint host details api */
  generateHostInfo(overrides = {}) {
    const hostInfo = {
      metadata: this.generate(),
      host_status: _types.HostStatus.HEALTHY,
      policy_info: {
        endpoint: {
          id: 'policy-123',
          revision: 4
        },
        agent: {
          applied: {
            id: 'policy-123',
            revision: 4
          },
          configured: {
            id: 'policy-123',
            revision: 4
          }
        }
      }
    };
    return (0, _lodash.merge)(hostInfo, overrides);
  }
  randomOsFields() {
    return this.randomChoice([EndpointMetadataGenerator.windowsOSFields, {
      name: 'Windows',
      full: 'Windows Server 2016',
      version: '10.0',
      platform: 'Windows',
      family: 'windows',
      Ext: {
        variant: 'Windows Server'
      }
    }, {
      name: 'Windows',
      full: 'Windows Server 2012',
      version: '6.2',
      platform: 'Windows',
      family: 'windows',
      Ext: {
        variant: 'Windows Server'
      }
    }, {
      name: 'Windows',
      full: 'Windows Server 2012R2',
      version: '6.3',
      platform: 'Windows',
      family: 'windows',
      Ext: {
        variant: 'Windows Server Release 2'
      }
    }, EndpointMetadataGenerator.linuxOSFields, EndpointMetadataGenerator.macOSFields]);
  }
}
exports.EndpointMetadataGenerator = EndpointMetadataGenerator;