"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerJobInfoRoutes = registerJobInfoRoutes;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/security-plugin/server");
var _util = require("util");
var _constants = require("../../../common/constants");
var _lib = require("../../lib");
var _lib2 = require("../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAIN_ENTRY = `${_constants.API_BASE_URL}/jobs`;
function registerJobInfoRoutes(reporting) {
  const setupDeps = reporting.getPluginSetupDeps();
  const {
    router
  } = setupDeps;
  const jobsQuery = (0, _lib2.jobsQueryFactory)(reporting);
  const registerGetList = () => {
    // list jobs in the queue, paginated
    const path = `${MAIN_ENTRY}/list`;
    router.get({
      path,
      validate: {
        query: _configSchema.schema.object({
          page: _configSchema.schema.string({
            defaultValue: '0'
          }),
          size: _configSchema.schema.string({
            defaultValue: '10'
          }),
          ids: _configSchema.schema.maybe(_configSchema.schema.string())
        })
      }
    }, (0, _lib2.authorizedUserPreRouting)(reporting, async (user, context, req, res) => {
      const counters = (0, _lib2.getCounters)(req.route.method, path, reporting.getUsageCounter());

      // ensure the async dependencies are loaded
      if (!context.reporting) {
        return (0, _lib2.handleUnavailable)(res);
      }
      const {
        management: {
          jobTypes = []
        }
      } = await reporting.getLicenseInfo();
      const {
        page: queryPage = '0',
        size: querySize = '10',
        ids: queryIds = null
      } = req.query;
      const page = parseInt(queryPage, 10) || 0;
      const size = Math.min(100, parseInt(querySize, 10) || 10);
      const jobIds = queryIds ? queryIds.split(',') : null;
      const results = await jobsQuery.list(jobTypes, user, page, size, jobIds);
      counters.usageCounter();
      return res.ok({
        body: results,
        headers: {
          'content-type': 'application/json'
        }
      });
    }));
  };
  const registerGetCount = () => {
    // return the count of all jobs in the queue
    const path = `${MAIN_ENTRY}/count`;
    router.get({
      path,
      validate: false
    }, (0, _lib2.authorizedUserPreRouting)(reporting, async (user, context, req, res) => {
      const counters = (0, _lib2.getCounters)(req.route.method, path, reporting.getUsageCounter());

      // ensure the async dependencies are loaded
      if (!context.reporting) {
        return (0, _lib2.handleUnavailable)(res);
      }
      const {
        management: {
          jobTypes = []
        }
      } = await reporting.getLicenseInfo();
      const count = await jobsQuery.count(jobTypes, user);
      counters.usageCounter();
      return res.ok({
        body: count.toString(),
        headers: {
          'content-type': 'text/plain'
        }
      });
    }));
  };
  const registerGetInfo = () => {
    // return some info about the job
    const path = `${MAIN_ENTRY}/info/{docId}`;
    router.get({
      path,
      validate: {
        params: _configSchema.schema.object({
          docId: _configSchema.schema.string({
            minLength: 2
          })
        })
      }
    }, (0, _lib2.authorizedUserPreRouting)(reporting, async (user, context, req, res) => {
      const counters = (0, _lib2.getCounters)(req.route.method, path, reporting.getUsageCounter());

      // ensure the async dependencies are loaded
      if (!context.reporting) {
        return res.custom({
          statusCode: 503
        });
      }
      const {
        docId
      } = req.params;
      return (0, _lib2.jobManagementPreRouting)(reporting, res, docId, user, counters, async doc => res.ok({
        body: doc,
        headers: {
          'content-type': 'application/json'
        }
      }));
    }));
  };
  const registerDownloadReport = () => {
    // trigger a download of the output from a job
    const path = `${MAIN_ENTRY}/download/{docId}`;
    router.get({
      path,
      validate: {
        params: _configSchema.schema.object({
          docId: _configSchema.schema.string({
            minLength: 3
          })
        })
      },
      options: {
        tags: [_server.ROUTE_TAG_CAN_REDIRECT]
      }
    }, (0, _lib2.authorizedUserPreRouting)(reporting, async (user, context, req, res) => {
      const counters = (0, _lib2.getCounters)(req.route.method, path, reporting.getUsageCounter());

      // ensure the async dependencies are loaded
      if (!context.reporting) {
        return (0, _lib2.handleUnavailable)(res);
      }
      const {
        docId
      } = req.params;
      return (0, _lib2.jobManagementPreRouting)(reporting, res, docId, user, counters, async doc => {
        const payload = await jobsQuery.getDocumentPayload(doc);
        const {
          contentType,
          content,
          filename,
          statusCode
        } = payload;
        if (!contentType || !_constants.ALLOWED_JOB_CONTENT_TYPES.includes(contentType)) {
          return res.badRequest({
            body: `Unsupported content-type of ${contentType} specified by job output`
          });
        }
        const body = typeof content === 'string' ? Buffer.from(content) : content;
        const headers = {
          ...payload.headers,
          'content-type': contentType
        };
        if (filename) {
          return res.file({
            body,
            headers,
            filename
          });
        }
        return res.custom({
          body,
          headers,
          statusCode
        });
      });
    }));
  };
  const registerDeleteReport = () => {
    // allow a report to be deleted
    const path = `${MAIN_ENTRY}/delete/{docId}`;
    router.delete({
      path,
      validate: {
        params: _configSchema.schema.object({
          docId: _configSchema.schema.string({
            minLength: 3
          })
        })
      }
    }, (0, _lib2.authorizedUserPreRouting)(reporting, async (user, context, req, res) => {
      const counters = (0, _lib2.getCounters)(req.route.method, path, reporting.getUsageCounter());

      // ensure the async dependencies are loaded
      if (!context.reporting) {
        return (0, _lib2.handleUnavailable)(res);
      }
      const {
        docId
      } = req.params;
      return (0, _lib2.jobManagementPreRouting)(reporting, res, docId, user, counters, async doc => {
        const docIndex = doc.index;
        const stream = await (0, _lib.getContentStream)(reporting, {
          id: docId,
          index: docIndex
        });

        /** @note Overwriting existing content with an empty buffer to remove all the chunks. */
        await (0, _util.promisify)(stream.end.bind(stream, '', 'utf8'))();
        await jobsQuery.delete(docIndex, docId);
        return res.ok({
          body: {
            deleted: true
          }
        });
      });
    }));
  };
  registerGetList();
  registerGetCount();
  registerGetInfo();
  registerDownloadReport();
  registerDeleteReport();
}