"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReportingNotifierStreamHandler = void 0;
var _i18n = require("@kbn/i18n");
var Rx = _interopRequireWildcard(require("rxjs"));
var _operators = require("rxjs/operators");
var _constants = require("../../common/constants");
var _notifier = require("../notifier");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * @todo Replace with `Infinity` once elastic/eui#5945 is resolved.
 * @see https://github.com/elastic/eui/issues/5945
 */
const COMPLETED_JOB_TOAST_TIMEOUT = 24 * 60 * 60 * 1000; // 24 hours

function updateStored(jobIds) {
  sessionStorage.setItem(_constants.JOB_COMPLETION_NOTIFICATIONS_SESSION_KEY, JSON.stringify(jobIds));
}
function getReportStatus(src) {
  var _src$prettyJobTypeNam;
  return {
    id: src.id,
    status: src.status,
    title: src.title,
    jobtype: (_src$prettyJobTypeNam = src.prettyJobTypeName) !== null && _src$prettyJobTypeNam !== void 0 ? _src$prettyJobTypeNam : src.jobtype,
    maxSizeReached: src.max_size_reached,
    csvContainsFormulas: src.csv_contains_formulas,
    errorCode: src.error_code
  };
}
class ReportingNotifierStreamHandler {
  constructor(notifications, apiClient, theme, docLinks) {
    this.notifications = notifications;
    this.apiClient = apiClient;
    this.theme = theme;
    this.docLinks = docLinks;
  }

  /*
   * Use Kibana Toast API to show our messages
   */
  showNotifications({
    completed: completedJobs,
    failed: failedJobs
  }) {
    const showNotificationsAsync = async () => {
      const completedOptions = {
        toastLifeTimeMs: COMPLETED_JOB_TOAST_TIMEOUT
      };

      // notifications with download link
      for (const job of completedJobs) {
        if (job.csvContainsFormulas) {
          this.notifications.toasts.addWarning((0, _notifier.getWarningFormulasToast)(job, this.apiClient.getManagementLink, this.apiClient.getDownloadLink, this.theme), completedOptions);
        } else if (job.maxSizeReached) {
          this.notifications.toasts.addWarning((0, _notifier.getWarningMaxSizeToast)(job, this.apiClient.getManagementLink, this.apiClient.getDownloadLink, this.theme), completedOptions);
        } else if (job.status === _constants.JOB_STATUSES.WARNINGS) {
          this.notifications.toasts.addWarning((0, _notifier.getWarningToast)(job, this.apiClient.getManagementLink, this.apiClient.getDownloadLink, this.theme), completedOptions);
        } else {
          this.notifications.toasts.addSuccess((0, _notifier.getSuccessToast)(job, this.apiClient.getManagementLink, this.apiClient.getDownloadLink, this.theme), completedOptions);
        }
      }

      // no download link available
      for (const job of failedJobs) {
        const errorText = await this.apiClient.getError(job.id);
        this.notifications.toasts.addDanger((0, _notifier.getFailureToast)(errorText, job, this.apiClient.getManagementLink, this.theme, this.docLinks));
      }
      return {
        completed: completedJobs,
        failed: failedJobs
      };
    };
    return Rx.from(showNotificationsAsync()); // convert Promise to Observable, for the convenience of the main stream
  }

  /*
   * An observable that finds jobs that are known to be "processing" (stored in
   * session storage) but have non-processing job status on the server
   */
  findChangedStatusJobs(storedJobs) {
    return Rx.from(this.apiClient.findForJobIds(storedJobs)).pipe((0, _operators.map)(jobs => {
      const completedJobs = [];
      const failedJobs = [];
      const pending = [];

      // add side effects to storage
      for (const job of jobs) {
        const {
          id: jobId,
          status: jobStatus
        } = job;
        if (storedJobs.includes(jobId)) {
          if (jobStatus === _constants.JOB_STATUSES.COMPLETED || jobStatus === _constants.JOB_STATUSES.WARNINGS) {
            completedJobs.push(getReportStatus(job));
          } else if (jobStatus === _constants.JOB_STATUSES.FAILED) {
            failedJobs.push(getReportStatus(job));
          } else {
            pending.push(jobId);
          }
        }
      }
      updateStored(pending); // refresh the storage of pending job IDs, minus completed and failed job IDs

      return {
        completed: completedJobs,
        failed: failedJobs
      };
    }), (0, _operators.catchError)(err => {
      // show connection refused toast
      this.notifications.toasts.addDanger((0, _notifier.getGeneralErrorToast)(_i18n.i18n.translate('xpack.reporting.publicNotifier.httpErrorMessage', {
        defaultMessage: 'Could not check Reporting job status!'
      }), err, this.theme)); // prettier-ignore
      window.console.error(err);
      return Rx.of({
        completed: [],
        failed: []
      }); // log the error and resume
    }));
  }
}
exports.ReportingNotifierStreamHandler = ReportingNotifierStreamHandler;