"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerFlameChartSearchRoute = registerFlameChartSearchRoute;
var _configSchema = require("@kbn/config-schema");
var _common = require("../../common");
var _callee = require("../../common/callee");
var _handle_route_error_handler = require("../utils/handle_route_error_handler");
var _flamegraph = require("../../common/flamegraph");
var _with_profiling_span = require("../utils/with_profiling_span");
var _compat = require("./compat");
var _query = require("./query");
var _search_stacktraces = require("./search_stacktraces");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerFlameChartSearchRoute({
  router,
  logger,
  services: {
    createProfilingEsClient
  }
}) {
  const paths = (0, _common.getRoutePaths)();
  router.get({
    path: paths.Flamechart,
    options: {
      tags: ['access:profiling']
    },
    validate: {
      query: _configSchema.schema.object({
        timeFrom: _configSchema.schema.number(),
        timeTo: _configSchema.schema.number(),
        kuery: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const {
      timeFrom,
      timeTo,
      kuery
    } = request.query;
    const targetSampleSize = 20000; // minimum number of samples to get statistically sound results

    try {
      const esClient = await (0, _compat.getClient)(context);
      const profilingElasticsearchClient = createProfilingEsClient({
        request,
        esClient
      });
      const filter = (0, _query.createCommonFilter)({
        timeFrom,
        timeTo,
        kuery
      });
      const totalSeconds = timeTo - timeFrom;
      const {
        stackTraceEvents,
        stackTraces,
        executables,
        stackFrames,
        totalFrames,
        samplingRate
      } = await (0, _search_stacktraces.searchStackTraces)({
        client: profilingElasticsearchClient,
        filter,
        sampleSize: targetSampleSize
      });
      const flamegraph = await (0, _with_profiling_span.withProfilingSpan)('create_flamegraph', async () => {
        const tree = (0, _callee.createCalleeTree)(stackTraceEvents, stackTraces, stackFrames, executables, totalFrames, samplingRate);
        const fg = (0, _flamegraph.createBaseFlameGraph)(tree, samplingRate, totalSeconds);
        return fg;
      });
      return response.ok({
        body: flamegraph
      });
    } catch (error) {
      return (0, _handle_route_error_handler.handleRouteHandlerError)({
        error,
        logger,
        response,
        message: 'Error while fetching flamegraph'
      });
    }
  });
}