"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MetricCustomTransformGenerator = exports.INVALID_EQUATION_REGEX = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _errors = require("../../../errors");
var _slo_transform_template = require("../../../assets/transform_templates/slo_transform_template");
var _ = require(".");
var _constants = require("../../../assets/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INVALID_EQUATION_REGEX = /[^A-Z|+|\-|\s|\d+|\.|\(|\)|\/|\*|>|<|=|\?|\:|&|\!|\|]+/g;
exports.INVALID_EQUATION_REGEX = INVALID_EQUATION_REGEX;
class MetricCustomTransformGenerator extends _.TransformGenerator {
  getTransformParams(slo) {
    if (!_sloSchema.metricCustomIndicatorSchema.is(slo.indicator)) {
      throw new _errors.InvalidTransformError(`Cannot handle SLO of indicator type: ${slo.indicator.type}`);
    }
    return (0, _slo_transform_template.getSLOTransformTemplate)(this.buildTransformId(slo), this.buildDescription(slo), this.buildSource(slo, slo.indicator), this.buildDestination(), this.buildGroupBy(slo, slo.indicator.params.timestampField), this.buildAggregations(slo, slo.indicator), this.buildSettings(slo, slo.indicator.params.timestampField));
  }
  buildTransformId(slo) {
    return (0, _constants.getSLOTransformId)(slo.id, slo.revision);
  }
  buildSource(slo, indicator) {
    const filter = (0, _.getElastichsearchQueryOrThrow)(indicator.params.filter);
    return {
      index: indicator.params.index,
      runtime_mappings: this.buildCommonRuntimeMappings(slo),
      query: filter
    };
  }
  buildDestination() {
    return {
      pipeline: _constants.SLO_INGEST_PIPELINE_NAME,
      index: _constants.SLO_DESTINATION_INDEX_NAME
    };
  }
  buildMetricAggregations(type, metricDef) {
    return metricDef.metrics.reduce((acc, metric) => ({
      ...acc,
      [`_${type}_${metric.name}`]: {
        [metric.aggregation]: {
          field: metric.field
        }
      }
    }), {});
  }
  convertEquationToPainless(bucketsPath, equation) {
    const workingEquation = equation || Object.keys(bucketsPath).join(' + ');
    return Object.keys(bucketsPath).reduce((acc, key) => {
      return acc.replace(key, `params.${key}`);
    }, workingEquation);
  }
  buildMetricEquation(type, metricDef) {
    const bucketsPath = metricDef.metrics.reduce((acc, metric) => ({
      ...acc,
      [metric.name]: `_${type}_${metric.name}`
    }), {});
    return {
      bucket_script: {
        buckets_path: bucketsPath,
        script: {
          source: this.convertEquationToPainless(bucketsPath, metricDef.equation),
          lang: 'painless'
        }
      }
    };
  }
  buildAggregations(slo, indicator) {
    if (indicator.params.good.equation.match(INVALID_EQUATION_REGEX)) {
      throw new Error(`Invalid equation: ${indicator.params.good.equation}`);
    }
    if (indicator.params.total.equation.match(INVALID_EQUATION_REGEX)) {
      throw new Error(`Invalid equation: ${indicator.params.total.equation}`);
    }
    const goodAggregations = this.buildMetricAggregations('good', indicator.params.good);
    const totalAggregations = this.buildMetricAggregations('total', indicator.params.total);
    return {
      ...goodAggregations,
      ...totalAggregations,
      'slo.numerator': this.buildMetricEquation('good', indicator.params.good),
      'slo.denominator': this.buildMetricEquation('total', indicator.params.total),
      ...(_sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod) && {
        'slo.isGoodSlice': {
          bucket_script: {
            buckets_path: {
              goodEvents: 'slo.numerator>value',
              totalEvents: 'slo.denominator>value'
            },
            script: `params.goodEvents / params.totalEvents >= ${slo.objective.timesliceTarget} ? 1 : 0`
          }
        }
      })
    };
  }
}
exports.MetricCustomTransformGenerator = MetricCustomTransformGenerator;