"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRuleExecutor = void 0;
var _i18n = require("@kbn/i18n");
var _numeral = _interopRequireDefault(require("@elastic/numeral"));
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _lodash = require("lodash");
var _server = require("@kbn/spaces-plugin/server");
var _common = require("../../../../common");
var _infra_metrics = require("../../../../common/field_names/infra_metrics");
var _models = require("../../../domain/models");
var _slo = require("../../../services/slo");
var _services = require("../../../domain/services");
var _types = require("./types");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SHORT_WINDOW = 'SHORT_WINDOW';
const LONG_WINDOW = 'LONG_WINDOW';
async function evaluateWindow(slo, summaryClient, windowDef) {
  const longWindowDuration = new _models.Duration(windowDef.longWindow.value, (0, _models.toDurationUnit)(windowDef.longWindow.unit));
  const shortWindowDuration = new _models.Duration(windowDef.shortWindow.value, (0, _models.toDurationUnit)(windowDef.shortWindow.unit));
  const sliData = await summaryClient.fetchSLIDataFrom(slo, [{
    name: LONG_WINDOW,
    duration: longWindowDuration.add(slo.settings.syncDelay)
  }, {
    name: SHORT_WINDOW,
    duration: shortWindowDuration.add(slo.settings.syncDelay)
  }]);
  const longWindowBurnRate = (0, _services.computeBurnRate)(slo, sliData[LONG_WINDOW]);
  const shortWindowBurnRate = (0, _services.computeBurnRate)(slo, sliData[SHORT_WINDOW]);
  const shouldAlert = longWindowBurnRate >= windowDef.burnRateThreshold && shortWindowBurnRate >= windowDef.burnRateThreshold;
  return {
    shouldAlert,
    longWindowBurnRate,
    shortWindowBurnRate,
    longWindowDuration,
    shortWindowDuration,
    window: windowDef
  };
}
async function evaluate(slo, summaryClient, params) {
  const evalWindow = (0, _lodash.memoize)(async windowDef => evaluateWindow(slo, summaryClient, windowDef));
  for (const windowDef of params.windows) {
    const result = await evalWindow(windowDef);
    if (result.shouldAlert) {
      return result;
    }
  }
  // If none of the previous windows match, we need to return the last window
  // for the recovery context. Since evalWindow is memoized, it shouldn't make
  // and additional call to evaulateWindow.
  return await evalWindow((0, _lodash.last)(params.windows));
}
const getRuleExecutor = ({
  basePath,
  alertsLocator
}) => async function executor({
  services,
  params,
  startedAt,
  spaceId
}) {
  const {
    alertWithLifecycle,
    savedObjectsClient: soClient,
    scopedClusterClient: esClient,
    alertFactory,
    getAlertStartedDate,
    getAlertUuid
  } = services;
  const sloRepository = new _slo.KibanaSavedObjectsSLORepository(soClient);
  const summaryClient = new _slo.DefaultSLIClient(esClient.asCurrentUser);
  const slo = await sloRepository.findById(params.sloId);
  if (!slo.enabled) {
    return {
      state: {}
    };
  }
  const result = await evaluate(slo, summaryClient, params);
  if (result) {
    const {
      shouldAlert,
      longWindowDuration,
      longWindowBurnRate,
      shortWindowDuration,
      shortWindowBurnRate,
      window: windowDef
    } = result;
    const viewInAppUrl = (0, _server.addSpaceIdToPath)(basePath.publicBaseUrl, spaceId, `/app/observability/slos/${slo.id}`);
    if (shouldAlert) {
      var _getAlertStartedDate;
      const reason = buildReason(windowDef.actionGroup, longWindowDuration, longWindowBurnRate, shortWindowDuration, shortWindowBurnRate, windowDef);
      const alertId = `alert-${slo.id}-${slo.revision}`;
      const indexedStartedAt = (_getAlertStartedDate = getAlertStartedDate(alertId)) !== null && _getAlertStartedDate !== void 0 ? _getAlertStartedDate : startedAt.toISOString();
      const alertUuid = getAlertUuid(alertId);
      const alertDetailsUrl = await (0, _common.getAlertUrl)(alertUuid, spaceId, indexedStartedAt, alertsLocator, basePath.publicBaseUrl);
      const context = {
        alertDetailsUrl,
        reason,
        longWindow: {
          burnRate: longWindowBurnRate,
          duration: longWindowDuration.format()
        },
        shortWindow: {
          burnRate: shortWindowBurnRate,
          duration: shortWindowDuration.format()
        },
        burnRateThreshold: windowDef.burnRateThreshold,
        timestamp: startedAt.toISOString(),
        viewInAppUrl,
        sloId: slo.id,
        sloName: slo.name
      };
      const alert = alertWithLifecycle({
        id: alertId,
        fields: {
          [_ruleDataUtils.ALERT_REASON]: reason,
          [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: windowDef.burnRateThreshold,
          [_ruleDataUtils.ALERT_EVALUATION_VALUE]: Math.min(longWindowBurnRate, shortWindowBurnRate),
          [_infra_metrics.SLO_ID_FIELD]: slo.id,
          [_infra_metrics.SLO_REVISION_FIELD]: slo.revision
        }
      });
      alert.scheduleActions(windowDef.actionGroup, context);
      alert.replaceState({
        alertState: _types.AlertStates.ALERT
      });
    }
    const {
      getRecoveredAlerts
    } = alertFactory.done();
    const recoveredAlerts = getRecoveredAlerts();
    for (const recoveredAlert of recoveredAlerts) {
      var _getAlertStartedDate2;
      const alertId = `alert-${slo.id}-${slo.revision}`;
      const indexedStartedAt = (_getAlertStartedDate2 = getAlertStartedDate(alertId)) !== null && _getAlertStartedDate2 !== void 0 ? _getAlertStartedDate2 : startedAt.toISOString();
      const alertUuid = getAlertUuid(alertId);
      const alertDetailsUrl = await (0, _common.getAlertUrl)(alertUuid, spaceId, indexedStartedAt, alertsLocator, basePath.publicBaseUrl);
      const context = {
        longWindow: {
          burnRate: longWindowBurnRate,
          duration: longWindowDuration.format()
        },
        shortWindow: {
          burnRate: shortWindowBurnRate,
          duration: shortWindowDuration.format()
        },
        burnRateThreshold: windowDef.burnRateThreshold,
        timestamp: startedAt.toISOString(),
        viewInAppUrl,
        alertDetailsUrl,
        sloId: slo.id,
        sloName: slo.name
      };
      recoveredAlert.setContext(context);
    }
  }
  return {
    state: {}
  };
};
exports.getRuleExecutor = getRuleExecutor;
function getActionGroupName(id) {
  switch (id) {
    case _constants.HIGH_PRIORITY_ACTION.id:
      return _constants.HIGH_PRIORITY_ACTION.name;
    case _constants.MEDIUM_PRIORITY_ACTION.id:
      return _constants.MEDIUM_PRIORITY_ACTION.name;
    case _constants.LOW_PRIORITY_ACTION.id:
      return _constants.LOW_PRIORITY_ACTION.name;
    default:
      return _constants.ALERT_ACTION.name;
  }
}
function buildReason(actionGroup, longWindowDuration, longWindowBurnRate, shortWindowDuration, shortWindowBurnRate, windowDef) {
  return _i18n.i18n.translate('xpack.observability.slo.alerting.burnRate.reason', {
    defaultMessage: '{actionGroupName}: The burn rate for the past {longWindowDuration} is {longWindowBurnRate} and for the past {shortWindowDuration} is {shortWindowBurnRate}. Alert when above {burnRateThreshold} for both windows',
    values: {
      actionGroupName: (0, _lodash.upperCase)(getActionGroupName(actionGroup)),
      longWindowDuration: longWindowDuration.format(),
      longWindowBurnRate: (0, _numeral.default)(longWindowBurnRate).format('0.[00]'),
      shortWindowDuration: shortWindowDuration.format(),
      shortWindowBurnRate: (0, _numeral.default)(shortWindowBurnRate).format('0.[00]'),
      burnRateThreshold: windowDef.burnRateThreshold
    }
  });
}