"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFetchCompositeSloList = useFetchCompositeSloList;
var _react = require("react");
var _reactQuery = require("@tanstack/react-query");
var _i18n = require("@kbn/i18n");
var _kibana_react = require("../../utils/kibana_react");
var _query_key_factory = require("./query_key_factory");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SHORT_REFETCH_INTERVAL = 1000 * 5; // 5 seconds
const LONG_REFETCH_INTERVAL = 1000 * 60; // 1 minute

function useFetchCompositeSloList({
  name = '',
  page = 1,
  sortBy = 'creationTime',
  shouldRefetch
} = {}) {
  const {
    http,
    notifications: {
      toasts
    }
  } = (0, _kibana_react.useKibana)().services;
  const queryClient = (0, _reactQuery.useQueryClient)();
  const [stateRefetchInterval, setStateRefetchInterval] = (0, _react.useState)(SHORT_REFETCH_INTERVAL);
  const {
    isInitialLoading,
    isLoading,
    isError,
    isSuccess,
    isRefetching,
    data,
    refetch
  } = (0, _reactQuery.useQuery)({
    queryKey: _query_key_factory.compositeSloKeys.list({
      name,
      page,
      sortBy
    }),
    queryFn: async ({
      signal
    }) => {
      try {
        const response = await http.get(`/api/observability/slos`, {
          query: {
            ...(page && {
              page
            }),
            ...(name && {
              name
            }),
            ...(sortBy && {
              sortBy
            })
          },
          signal
        });
        return response;
      } catch (error) {
        throw error;
      }
    },
    keepPreviousData: true,
    refetchOnWindowFocus: false,
    refetchInterval: shouldRefetch ? stateRefetchInterval : undefined,
    staleTime: 1000,
    retry: (failureCount, error) => {
      if (String(error) === 'Error: Forbidden') {
        return false;
      }
      return failureCount < 4;
    },
    onSuccess: ({
      results
    }) => {
      if (!shouldRefetch) {
        return;
      }
      if (results.find(slo => slo.summary.status === 'NO_DATA' || !slo.summary)) {
        setStateRefetchInterval(SHORT_REFETCH_INTERVAL);
      } else {
        setStateRefetchInterval(LONG_REFETCH_INTERVAL);
      }
      queryClient.invalidateQueries(_query_key_factory.compositeSloKeys.historicalSummaries());
      queryClient.invalidateQueries(_query_key_factory.compositeSloKeys.activeAlerts());
      queryClient.invalidateQueries(_query_key_factory.compositeSloKeys.rules());
    },
    onError: error => {
      toasts.addError(error, {
        title: _i18n.i18n.translate('xpack.observability.slo.list.errorNotification', {
          defaultMessage: 'Something went wrong while fetching SLOs'
        })
      });
    }
  });
  return {
    sloList: data,
    isInitialLoading,
    isLoading,
    isRefetching,
    isSuccess,
    isError,
    refetch
  };
}