"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.QuickGeoJobCreator = void 0;
var _i18n = require("@kbn/i18n");
var _dependency_cache = require("../../../util/dependency_cache");
var _new_job = require("../../../../../common/constants/new_job");
var _default_configs = require("../common/job_creator/util/default_configs");
var _general = require("../common/job_creator/util/general");
var _utils = require("./utils");
var _job_from_dashboard = require("../job_from_dashboard");
var _new_job_utils = require("../utils/new_job_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class QuickGeoJobCreator extends _job_from_dashboard.QuickJobCreatorBase {
  constructor(kibanaConfig, timeFilter, share, mlApiServices) {
    super(kibanaConfig, timeFilter, share, mlApiServices);
    this.kibanaConfig = kibanaConfig;
    this.timeFilter = timeFilter;
    this.share = share;
    this.mlApiServices = mlApiServices;
  }
  async createAndSaveGeoJob({
    jobId,
    bucketSpan,
    embeddable,
    startJob,
    runInRealTime,
    dataViewId,
    sourceDataView,
    geoField,
    splitField,
    layerLevelQuery
  }) {
    var _await$embeddable$get, _await$embeddable$get2;
    const {
      query: dashboardQuery,
      filters: dashboardFilters,
      to,
      from,
      dashboard
    } = await (0, _utils.getJobsItemsFromEmbeddable)(embeddable);

    // Map level stuff
    const embeddableQuery = (_await$embeddable$get = await embeddable.getQuery()) !== null && _await$embeddable$get !== void 0 ? _await$embeddable$get : (0, _new_job_utils.getDefaultQuery)();
    const embeddableFilters = (_await$embeddable$get2 = await embeddable.getFilters()) !== null && _await$embeddable$get2 !== void 0 ? _await$embeddable$get2 : [];
    if (dashboardQuery === undefined || dashboardFilters === undefined) {
      throw new Error('Cannot create job, query and filters are undefined');
    }
    const {
      jobConfig,
      datafeedConfig,
      start,
      end
    } = await this.createGeoJob({
      dataViewId,
      sourceDataView,
      from,
      to,
      query: dashboardQuery,
      filters: dashboardFilters,
      embeddableQuery,
      embeddableFilters,
      layerLevelQuery,
      geoField,
      splitField,
      bucketSpan
    });
    const result = await this.putJobAndDataFeed({
      jobId,
      datafeedConfig,
      jobConfig,
      createdByLabel: _new_job.CREATED_BY_LABEL.GEO,
      dashboard,
      start,
      end,
      startJob,
      runInRealTime
    });
    return result;
  }
  async createAndStashGeoJob(dataViewId, startString, endString, query, filters, embeddableQuery, embeddableFilters, geoField, splitField = null, layerLevelQuery) {
    try {
      const {
        jobConfig,
        datafeedConfig,
        start,
        end,
        includeTimeRange
      } = await this.createGeoJob({
        dataViewId,
        from: startString,
        to: endString,
        query,
        filters,
        embeddableQuery,
        embeddableFilters,
        geoField,
        splitField,
        layerLevelQuery
      });

      // add job config and start and end dates to the
      // job cloning stash, so they can be used
      // by the new job wizards
      (0, _general.stashJobForCloning)({
        jobConfig,
        datafeedConfig,
        createdBy: _new_job.CREATED_BY_LABEL.GEO,
        start,
        end
      }, true, includeTimeRange, !includeTimeRange);
    } catch (error) {
      // eslint-disable-next-line no-console
      console.error(error);
    }
  }
  async createGeoJob({
    dataViewId,
    sourceDataView,
    from,
    to,
    query,
    filters,
    embeddableQuery,
    embeddableFilters,
    layerLevelQuery,
    geoField,
    splitField,
    bucketSpan
  }) {
    const {
      jobConfig,
      datafeedConfig,
      jobType
    } = await this.createGeoJobFromMapEmbeddable({
      sourceDataView,
      dataViewId,
      dashboard: {
        query,
        filters
      },
      embeddable: {
        query: embeddableQuery,
        filters: embeddableFilters
      },
      layerLevelQuery,
      geoField,
      splitField,
      ...(bucketSpan ? {
        bucketSpan
      } : {})
    });
    let start;
    let end;
    let includeTimeRange = true;
    try {
      // attempt to parse the start and end dates.
      // if start and end values cannot be determined
      // instruct the job cloning code to auto-select the
      // full time range for the index.
      const {
        min,
        max
      } = this.timeFilter.calculateBounds({
        to,
        from
      });
      start = min === null || min === void 0 ? void 0 : min.valueOf();
      end = max === null || max === void 0 ? void 0 : max.valueOf();
      if (start === undefined || end === undefined || isNaN(start) || isNaN(end)) {
        throw Error(_i18n.i18n.translate('xpack.ml.newJob.fromLens.createJob.error.timeRange', {
          defaultMessage: 'Incompatible time range'
        }));
      }
    } catch (error) {
      // eslint-disable-next-line no-console
      console.error(error);
      includeTimeRange = false;
      start = undefined;
      end = undefined;
    }
    return {
      jobConfig,
      datafeedConfig,
      jobType,
      start,
      end,
      includeTimeRange
    };
  }
  async createGeoJobFromMapEmbeddable({
    sourceDataView,
    dataViewId,
    dashboard,
    embeddable,
    layerLevelQuery,
    bucketSpan,
    geoField,
    splitField
  }) {
    const dataView = sourceDataView ? sourceDataView : await (0, _dependency_cache.getDataViews)().get(dataViewId, true);
    const jobConfig = (0, _default_configs.createEmptyJob)();
    const datafeedConfig = (0, _default_configs.createEmptyDatafeed)(dataView.getIndexPattern());
    const combinedFiltersAndQueries = this.combineQueriesAndFilters(dashboard, embeddable, dataView, layerLevelQuery ? {
      query: layerLevelQuery,
      filters: []
    } : undefined);
    datafeedConfig.query = combinedFiltersAndQueries;
    jobConfig.analysis_config.detectors = [{
      function: 'lat_long',
      field_name: geoField,
      ...(splitField ? {
        partition_field_name: splitField
      } : {})
    }];
    jobConfig.data_description.time_field = dataView.timeFieldName;
    jobConfig.analysis_config.bucket_span = bucketSpan !== null && bucketSpan !== void 0 ? bucketSpan : _new_job.DEFAULT_BUCKET_SPAN;
    if (splitField) {
      jobConfig.analysis_config.influencers = [splitField];
    }
    return {
      jobConfig,
      datafeedConfig,
      jobType: _new_job.JOB_TYPE.GEO
    };
  }
}
exports.QuickGeoJobCreator = QuickGeoJobCreator;