"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MlCapabilitiesService = void 0;
exports.checkCreateJobsCapabilitiesResolver = checkCreateJobsCapabilitiesResolver;
exports.checkGetManagementMlJobsResolver = checkGetManagementMlJobsResolver;
exports.checkPermission = checkPermission;
exports.createPermissionFailureMessage = createPermissionFailureMessage;
exports.usePermissionCheck = usePermissionCheck;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _i18n = require("@kbn/i18n");
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
var _lodash = require("lodash");
var _useObservable = _interopRequireDefault(require("react-use/lib/useObservable"));
var _react = require("react");
var _kibana = require("../contexts/kibana");
var _license = require("../license");
var _capabilities2 = require("../../../common/types/capabilities");
var _get_capabilities = require("./get_capabilities");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

let _capabilities = (0, _capabilities2.getDefaultCapabilities)();
const CAPABILITIES_REFRESH_INTERVAL = 60000;
class MlCapabilitiesService {
  /**
   * Updates on manual request, e.g. in the route resolver.
   * @private
   */

  constructor(mlApiServices) {
    (0, _defineProperty2.default)(this, "_isLoading$", new _rxjs.BehaviorSubject(true));
    (0, _defineProperty2.default)(this, "_updateRequested$", new _rxjs.BehaviorSubject(Date.now()));
    (0, _defineProperty2.default)(this, "_capabilities$", new _rxjs.BehaviorSubject(null));
    (0, _defineProperty2.default)(this, "capabilities$", this._capabilities$.pipe((0, _operators.distinctUntilChanged)(_lodash.isEqual)));
    (0, _defineProperty2.default)(this, "_subscription", void 0);
    this.mlApiServices = mlApiServices;
    this.init();
  }
  init() {
    this._subscription = (0, _rxjs.combineLatest)([this._updateRequested$, (0, _rxjs.timer)(0, CAPABILITIES_REFRESH_INTERVAL)]).pipe((0, _operators.tap)(() => {
      this._isLoading$.next(true);
    }), (0, _operators.switchMap)(() => (0, _rxjs.from)(this.mlApiServices.checkMlCapabilities())), (0, _operators.retry)({
      delay: CAPABILITIES_REFRESH_INTERVAL
    })).subscribe(results => {
      this._capabilities$.next(results.capabilities);
      this._isLoading$.next(false);

      /**
       * To support legacy use of {@link checkPermission}
       */
      _capabilities = results.capabilities;
    });
  }
  getCapabilities() {
    return this._capabilities$.getValue();
  }
  refreshCapabilities() {
    this._updateRequested$.next(Date.now());
  }
  destroy() {
    if (this._subscription) {
      this._subscription.unsubscribe();
    }
  }
}

/**
 * Check the privilege type and the license to see whether a user has permission to access a feature.
 *
 * @param capability
 */
exports.MlCapabilitiesService = MlCapabilitiesService;
function usePermissionCheck(capability) {
  const {
    services: {
      mlServices: {
        mlCapabilities: mlCapabilitiesService
      }
    }
  } = (0, _kibana.useMlKibana)();

  // Memoize argument, in case it's an array to preserve the reference.
  const requestedCapabilities = (0, _react.useRef)(capability);
  const capabilities = (0, _useObservable.default)(mlCapabilitiesService.capabilities$, mlCapabilitiesService.getCapabilities());
  return (0, _react.useMemo)(() => {
    return Array.isArray(requestedCapabilities.current) ? requestedCapabilities.current.map(c => capabilities[c]) : capabilities[requestedCapabilities.current];
  }, [capabilities]);
}
function checkGetManagementMlJobsResolver({
  checkMlCapabilities
}) {
  return new Promise((resolve, reject) => {
    checkMlCapabilities().then(({
      capabilities,
      isPlatinumOrTrialLicense,
      mlFeatureEnabledInSpace
    }) => {
      _capabilities = capabilities;
      // Loop through all capabilities to ensure they are all set to true.
      const isManageML = Object.values(_capabilities).every(p => p === true);
      if (isManageML === true && isPlatinumOrTrialLicense === true) {
        return resolve({
          mlFeatureEnabledInSpace
        });
      } else {
        return reject({
          capabilities,
          isPlatinumOrTrialLicense,
          mlFeatureEnabledInSpace
        });
      }
    }).catch(e => {
      return reject();
    });
  });
}
function checkCreateJobsCapabilitiesResolver(redirectToJobsManagementPage) {
  return new Promise((resolve, reject) => {
    (0, _get_capabilities.getCapabilities)().then(async ({
      capabilities,
      isPlatinumOrTrialLicense
    }) => {
      _capabilities = capabilities;
      // if the license is basic (isPlatinumOrTrialLicense === false) then do not redirect,
      // allow the promise to resolve as the separate license check will redirect then user to
      // a basic feature
      if (_capabilities.canCreateJob || isPlatinumOrTrialLicense === false) {
        return resolve(_capabilities);
      } else {
        // if the user has no permission to create a job,
        // redirect them back to the Anomaly Detection Management page
        await redirectToJobsManagementPage();
        return reject();
      }
    }).catch(async e => {
      await redirectToJobsManagementPage();
      return reject();
    });
  });
}

/**
 * @deprecated use {@link usePermissionCheck} instead.
 * @param capability
 */
function checkPermission(capability) {
  const licenseHasExpired = (0, _license.hasLicenseExpired)();
  return _capabilities[capability] === true && licenseHasExpired !== true;
}

// create the text for the button's tooltips if the user's license has
// expired or if they don't have the privilege to press that button
function createPermissionFailureMessage(privilegeType) {
  let message = '';
  const licenseHasExpired = (0, _license.hasLicenseExpired)();
  if (licenseHasExpired) {
    message = _i18n.i18n.translate('xpack.ml.privilege.licenseHasExpiredTooltip', {
      defaultMessage: 'Your license has expired.'
    });
  } else if (privilegeType === 'canCreateJob') {
    message = _i18n.i18n.translate('xpack.ml.privilege.noPermission.createMLJobsTooltip', {
      defaultMessage: 'You do not have permission to create Machine Learning jobs.'
    });
  } else if (privilegeType === 'canStartStopDatafeed') {
    message = _i18n.i18n.translate('xpack.ml.privilege.noPermission.startOrStopDatafeedsTooltip', {
      defaultMessage: 'You do not have permission to start or stop datafeeds.'
    });
  } else if (privilegeType === 'canUpdateJob') {
    message = _i18n.i18n.translate('xpack.ml.privilege.noPermission.editJobsTooltip', {
      defaultMessage: 'You do not have permission to edit jobs.'
    });
  } else if (privilegeType === 'canDeleteJob') {
    message = _i18n.i18n.translate('xpack.ml.privilege.noPermission.deleteJobsTooltip', {
      defaultMessage: 'You do not have permission to delete jobs.'
    });
  } else if (privilegeType === 'canCreateCalendar') {
    message = _i18n.i18n.translate('xpack.ml.privilege.noPermission.createCalendarsTooltip', {
      defaultMessage: 'You do not have permission to create calendars.'
    });
  } else if (privilegeType === 'canDeleteCalendar') {
    message = _i18n.i18n.translate('xpack.ml.privilege.noPermission.deleteCalendarsTooltip', {
      defaultMessage: 'You do not have permission to delete calendars.'
    });
  } else if (privilegeType === 'canForecastJob') {
    message = _i18n.i18n.translate('xpack.ml.privilege.noPermission.runForecastsTooltip', {
      defaultMessage: 'You do not have permission to run forecasts.'
    });
  }
  return _i18n.i18n.translate('xpack.ml.privilege.pleaseContactAdministratorTooltip', {
    defaultMessage: '{message} Please contact your administrator.',
    values: {
      message
    }
  });
}