"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DEFAULT_MAP_STATE = void 0;
exports.map = map;
var _constants = require("../../../common/constants");
var _map_action_constants = require("../../actions/map_action_constants");
var _default_map_settings = require("./default_map_settings");
var _layer_utils = require("./layer_utils");
var _data_request_utils = require("./data_request_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_MAP_STATE = {
  executionContext: {
    name: _constants.APP_ID
  },
  ready: false,
  mapInitError: null,
  goto: null,
  openTooltips: [],
  mapState: {
    zoom: undefined,
    // setting this value does not adjust map zoom, read only value used to store current map zoom for persisting between sessions
    center: undefined,
    // setting this value does not adjust map view, read only value used to store current map center for persisting between sessions
    extent: undefined,
    mouseCoordinates: undefined,
    timeFilters: undefined,
    timeslice: undefined,
    query: undefined,
    filters: [],
    drawState: undefined,
    editState: undefined
  },
  selectedLayerId: null,
  layerList: [],
  waitingForMapReadyLayerList: [],
  settings: (0, _default_map_settings.getDefaultMapSettings)(),
  __rollbackSettings: null
};
exports.DEFAULT_MAP_STATE = DEFAULT_MAP_STATE;
function map(state = DEFAULT_MAP_STATE, action) {
  switch (action.type) {
    case _map_action_constants.UPDATE_DRAW_STATE:
      return {
        ...state,
        mapState: {
          ...state.mapState,
          drawState: action.drawState
        }
      };
    case _map_action_constants.UPDATE_EDIT_STATE:
      return {
        ...state,
        mapState: {
          ...state.mapState,
          editState: action.editState
        }
      };
    case _map_action_constants.REMOVE_TRACKED_LAYER_STATE:
      return (0, _layer_utils.removeTrackedLayerState)(state, action.layerId);
    case _map_action_constants.TRACK_CURRENT_LAYER_STATE:
      return (0, _layer_utils.trackCurrentLayerState)(state, action.layerId);
    case _map_action_constants.ROLLBACK_TO_TRACKED_LAYER_STATE:
      return (0, _layer_utils.rollbackTrackedLayerState)(state, action.layerId);
    case _map_action_constants.SET_OPEN_TOOLTIPS:
      return {
        ...state,
        openTooltips: action.openTooltips
      };
    case _map_action_constants.SET_MOUSE_COORDINATES:
      return {
        ...state,
        mapState: {
          ...state.mapState,
          mouseCoordinates: {
            lat: action.lat,
            lon: action.lon
          }
        }
      };
    case _map_action_constants.CLEAR_MOUSE_COORDINATES:
      return {
        ...state,
        mapState: {
          ...state.mapState,
          mouseCoordinates: undefined
        }
      };
    case _map_action_constants.SET_GOTO:
      return {
        ...state,
        goto: {
          center: action.center,
          bounds: action.bounds
        }
      };
    case _map_action_constants.CLEAR_GOTO:
      return {
        ...state,
        goto: null
      };
    case _map_action_constants.SET_MAP_SETTINGS:
      return {
        ...state,
        settings: {
          ...state.settings,
          ...action.settings
        }
      };
    case _map_action_constants.ROLLBACK_MAP_SETTINGS:
      return state.__rollbackSettings ? {
        ...state,
        settings: {
          ...state.__rollbackSettings
        },
        __rollbackSettings: null
      } : state;
    case _map_action_constants.TRACK_MAP_SETTINGS:
      return {
        ...state,
        __rollbackSettings: state.settings
      };
    case _map_action_constants.UPDATE_MAP_SETTING:
      return {
        ...state,
        settings: {
          ...(state.settings ? state.settings : {}),
          [action.settingKey]: action.settingValue
        }
      };
    case _map_action_constants.SET_LAYER_ERROR_STATUS:
      const {
        layerList
      } = state;
      const layerIdx = (0, _layer_utils.getLayerIndex)(layerList, action.layerId);
      if (layerIdx === -1) {
        return state;
      }
      return {
        ...state,
        layerList: [...layerList.slice(0, layerIdx), {
          ...layerList[layerIdx],
          __isInErrorState: action.isInErrorState,
          __errorMessage: action.errorMessage
        }, ...layerList.slice(layerIdx + 1)]
      };
    case _map_action_constants.UPDATE_SOURCE_DATA_REQUEST:
      return (0, _data_request_utils.updateSourceDataRequest)(state, action.layerId, action.newData);
    case _map_action_constants.LAYER_DATA_LOAD_STARTED:
      return (0, _data_request_utils.startDataRequest)(state, action.layerId, action.dataId, action.requestToken, action.meta);
    case _map_action_constants.LAYER_DATA_LOAD_ERROR:
      return (0, _data_request_utils.stopDataRequest)(state, action.layerId, action.dataId, action.requestToken);
    case _map_action_constants.LAYER_DATA_LOAD_ENDED:
      return (0, _data_request_utils.stopDataRequest)(state, action.layerId, action.dataId, action.requestToken, action.meta, action.data);
    case _map_action_constants.MAP_READY:
      return {
        ...state,
        ready: true
      };
    case _map_action_constants.MAP_DESTROYED:
      return {
        ...state,
        ready: false
      };
    case _map_action_constants.MAP_EXTENT_CHANGED:
      return {
        ...state,
        mapState: {
          ...state.mapState,
          ...action.mapViewContext
        }
      };
    case _map_action_constants.SET_QUERY:
      const {
        query,
        timeFilters,
        timeslice,
        filters,
        searchSessionId,
        searchSessionMapBuffer
      } = action;
      return {
        ...state,
        mapState: {
          ...state.mapState,
          query,
          timeFilters,
          timeslice,
          filters,
          searchSessionId,
          searchSessionMapBuffer
        }
      };
    case _map_action_constants.SET_SELECTED_LAYER:
      const selectedMatch = state.layerList.find(layer => layer.id === action.selectedLayerId);
      return {
        ...state,
        selectedLayerId: selectedMatch ? action.selectedLayerId : null
      };
    case _map_action_constants.UPDATE_LAYER_ORDER:
      return {
        ...state,
        layerList: action.newLayerOrder.map(layerNumber => state.layerList[layerNumber])
      };
    case _map_action_constants.UPDATE_LAYER_PROP:
      return (0, _layer_utils.updateLayerInList)(state, action.id, action.propName, action.newValue);
    case _map_action_constants.CLEAR_LAYER_PROP:
      return (0, _layer_utils.clearLayerProp)(state, action.id, action.propName);
    case _map_action_constants.UPDATE_SOURCE_PROP:
      return (0, _layer_utils.updateLayerSourceDescriptorProp)(state, action.layerId, action.propName, action.value);
    case _map_action_constants.SET_JOINS:
      const layerDescriptor = state.layerList.find(descriptor => descriptor.id === action.layerId);
      if (layerDescriptor) {
        const newLayerDescriptor = {
          ...layerDescriptor,
          joins: action.joins.slice()
        };
        const index = state.layerList.findIndex(descriptor => descriptor.id === action.layerId);
        const newLayerList = state.layerList.slice();
        newLayerList[index] = newLayerDescriptor;
        return {
          ...state,
          layerList: newLayerList
        };
      }
      return state;
    case _map_action_constants.ADD_LAYER:
      return {
        ...state,
        layerList: [...state.layerList, action.layer]
      };
    case _map_action_constants.REMOVE_LAYER:
      return {
        ...state,
        layerList: [...state.layerList.filter(({
          id
        }) => id !== action.id)]
      };
    case _map_action_constants.UPDATE_LAYER:
      return {
        ...state,
        layerList: (0, _layer_utils.setLayer)(state.layerList, action.layer)
      };
    case _map_action_constants.ADD_WAITING_FOR_MAP_READY_LAYER:
      return {
        ...state,
        waitingForMapReadyLayerList: [...state.waitingForMapReadyLayerList, action.layer]
      };
    case _map_action_constants.CLEAR_WAITING_FOR_MAP_READY_LAYER_LIST:
      return {
        ...state,
        waitingForMapReadyLayerList: []
      };
    case _map_action_constants.SET_LAYER_VISIBILITY:
      return (0, _layer_utils.updateLayerInList)(state, action.layerId, 'visible', action.visibility);
    case _map_action_constants.UPDATE_LAYER_STYLE:
      const styleLayerId = action.layerId;
      return (0, _layer_utils.updateLayerInList)(state, styleLayerId, 'style', {
        ...action.style
      });
    case _map_action_constants.SET_LAYER_STYLE_META:
      const {
        layerId,
        styleMeta
      } = action;
      const index = (0, _layer_utils.getLayerIndex)(state.layerList, layerId);
      if (index === -1) {
        return state;
      }
      return (0, _layer_utils.updateLayerInList)(state, layerId, 'style', {
        ...state.layerList[index].style,
        __styleMeta: styleMeta
      });
    case _map_action_constants.SET_MAP_INIT_ERROR:
      return {
        ...state,
        mapInitError: action.errorMessage
      };
    case _map_action_constants.SET_WAITING_FOR_READY_HIDDEN_LAYERS:
      return {
        ...state,
        waitingForMapReadyLayerList: state.waitingForMapReadyLayerList.map(layer => ({
          ...layer,
          visible: !action.hiddenLayerIds.includes(layer.id)
        }))
      };
    case _map_action_constants.SET_EMBEDDABLE_SEARCH_CONTEXT:
      return {
        ...state,
        mapState: {
          ...state.mapState,
          embeddableSearchContext: action.embeddableSearchContext
        }
      };
    case _map_action_constants.SET_EXECUTION_CONTEXT:
      {
        return {
          ...state,
          executionContext: action.executionContext
        };
      }
    default:
      return state;
  }
}