"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getOriginalRequestErrorMessages = getOriginalRequestErrorMessages;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _public = require("@kbn/data-plugin/public");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isNetworkError = e => {
  return e.message === 'Batch request failed with status 0'; // Note: 0 here means Network error
};

const isRequestError = e => {
  if ('body' in e) {
    var _e$body, _e$body$attributes, _e$body$attributes$er;
    return ((_e$body = e.body) === null || _e$body === void 0 ? void 0 : (_e$body$attributes = _e$body.attributes) === null || _e$body$attributes === void 0 ? void 0 : (_e$body$attributes$er = _e$body$attributes.error) === null || _e$body$attributes$er === void 0 ? void 0 : _e$body$attributes$er.caused_by) !== undefined;
  }
  return false;
};

// what happens for runtime field used on indexpatterns not accessible to the user?
// they will throw on the kibana side as data will be undefined
const isEsAggError = e => {
  return 'message' in e && 'stack' in e && !isRequestError(e) && !(0, _public.isEsError)(e);
};
function getNestedErrorClauseWithContext({
  type,
  reason = '',
  caused_by: causedBy,
  lang,
  script
}) {
  if (!causedBy) {
    // scripted fields error has changed with no particular hint about painless in it,
    // so it tries to lookup in the message for the script word
    if (/script/.test(reason)) {
      return [{
        type,
        reason,
        context: {
          type: 'Painless script',
          reason: ''
        }
      }];
    }
    return [{
      type,
      reason
    }];
  }
  const [payload] = getNestedErrorClause(causedBy);
  if (lang === 'painless') {
    return [{
      ...payload,
      context: {
        type: 'Painless script',
        reason: `"${script}"` || reason
      }
    }];
  }
  return [{
    ...payload,
    context: {
      type,
      reason
    }
  }];
}
function getNestedErrorClause(e) {
  const {
    type,
    reason = '',
    caused_by: causedBy
  } = e;
  // Painless scripts errors are nested within the failed_shards property
  if ('failed_shards' in e) {
    if (e.failed_shards) {
      return e.failed_shards.flatMap(shardCause => getNestedErrorClauseWithContext(shardCause.reason));
    }
  }
  if (causedBy) {
    return getNestedErrorClause(causedBy);
  }
  return [{
    type,
    reason
  }];
}
function getErrorSources(e) {
  if (isRequestError(e)) {
    return getNestedErrorClause(e.body.attributes.error);
  }
  if ((0, _public.isEsError)(e)) {
    var _e$attributes, _e$attributes2;
    if ((_e$attributes = e.attributes) !== null && _e$attributes !== void 0 && _e$attributes.reason) {
      return getNestedErrorClause(e.attributes);
    }
    return getNestedErrorClause((_e$attributes2 = e.attributes) === null || _e$attributes2 === void 0 ? void 0 : _e$attributes2.caused_by);
  }
  return [];
}
function getOriginalRequestErrorMessages(error) {
  const errorMessages = [];
  if (error && 'original' in error && error.original) {
    if (isEsAggError(error.original)) {
      if (isNetworkError(error.original)) {
        errorMessages.push(_i18n.i18n.translate('xpack.lens.editorFrame.networkErrorMessage', {
          defaultMessage: 'Network error, try again later or contact your administrator.'
        }));
      } else {
        errorMessages.push(error.message);
      }
    } else {
      const rootErrors = (0, _lodash.uniqWith)(getErrorSources(error.original), _lodash.isEqual);
      for (const rootError of rootErrors) {
        if (rootError.context) {
          errorMessages.push(_i18n.i18n.translate('xpack.lens.editorFrame.expressionFailureMessageWithContext', {
            defaultMessage: 'Request error: {type}, {reason} in {context}',
            values: {
              reason: rootError.reason,
              type: rootError.type,
              context: rootError.context.reason ? `${rootError.context.reason} (${rootError.context.type})` : rootError.context.type
            }
          }));
        } else {
          errorMessages.push(_i18n.i18n.translate('xpack.lens.editorFrame.expressionFailureMessage', {
            defaultMessage: 'Request error: {type}, {reason}',
            values: {
              reason: rootError.reason,
              type: rootError.type
            }
          }));
        }
      }
    }
  } else if (error !== null && error !== void 0 && error.message) {
    errorMessages.push(error === null || error === void 0 ? void 0 : error.message);
  }
  return errorMessages;
}

// NOTE - if you are adding a new error message, add it as a UserMessage in get_application_error_messages
// or the getUserMessages method of a particular datasource or visualization class! Alternatively, use the
// addUserMessage function passed down by the application component.