"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SUGGESTION_TYPE = exports.MARKER = void 0;
exports.createEditOperation = createEditOperation;
exports.getHover = getHover;
exports.getInfoAtZeroIndexedPosition = getInfoAtZeroIndexedPosition;
exports.getNamedArgumentSuggestions = getNamedArgumentSuggestions;
exports.getPossibleFunctions = getPossibleFunctions;
exports.getSignatureHelp = getSignatureHelp;
exports.getSuggestion = getSuggestion;
exports.getTokenInfo = getTokenInfo;
exports.monacoPositionToOffset = monacoPositionToOffset;
exports.offsetToRowColumn = offsetToRowColumn;
exports.suggest = suggest;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _monaco = require("@kbn/monaco");
var _tinymath = require("@kbn/tinymath");
var _common = require("@kbn/data-plugin/common");
var _moment = _interopRequireDefault(require("moment"));
var _utils = require("../../../../../../utils");
var _operations = require("../../../operations");
var _util = require("../util");
var _formula_help = require("./formula_help");
var _validation = require("../validation");
var _time_shift_utils = require("../../../../time_shift_utils");
var _reduced_time_range_utils = require("../../../../reduced_time_range_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
let SUGGESTION_TYPE;
exports.SUGGESTION_TYPE = SUGGESTION_TYPE;
(function (SUGGESTION_TYPE) {
  SUGGESTION_TYPE["FIELD"] = "field";
  SUGGESTION_TYPE["NAMED_ARGUMENT"] = "named_argument";
  SUGGESTION_TYPE["FUNCTIONS"] = "functions";
  SUGGESTION_TYPE["KQL"] = "kql";
  SUGGESTION_TYPE["SHIFTS"] = "shifts";
  SUGGESTION_TYPE["REDUCED_TIME_RANGES"] = "reducedTimeRanges";
})(SUGGESTION_TYPE || (exports.SUGGESTION_TYPE = SUGGESTION_TYPE = {}));
function inLocation(cursorPosition, location) {
  return cursorPosition >= location.min && cursorPosition < location.max;
}
const MARKER = 'LENS_MATH_MARKER';
exports.MARKER = MARKER;
function getInfoAtZeroIndexedPosition(ast, zeroIndexedPosition, parent) {
  if (typeof ast === 'number') {
    return;
  }
  // +, -, *, and / do not have location any more
  if (ast.location && !inLocation(zeroIndexedPosition, ast.location)) {
    return;
  }
  if (ast.type === 'function') {
    const [match] = ast.args.map(arg => getInfoAtZeroIndexedPosition(arg, zeroIndexedPosition, ast)).filter(_utils.nonNullable);
    if (match) {
      return match;
    } else if (ast.location) {
      return {
        ast
      };
    } else {
      // None of the arguments match, but we don't know the position so it's not a match
      return;
    }
  }
  return {
    ast,
    parent
  };
}
function createEditOperation(textToInject, currentPosition, startOffset = 0, endOffset = 1) {
  return {
    range: {
      ...currentPosition,
      // Insert after the current char
      startColumn: currentPosition.startColumn + startOffset,
      endColumn: currentPosition.startColumn + endOffset
    },
    text: textToInject
  };
}
function offsetToRowColumn(expression, offset) {
  const lines = expression.split(/\n/);
  let remainingChars = offset;
  let lineNumber = 1;
  for (const line of lines) {
    if (line.length >= remainingChars) {
      return new _monaco.monaco.Position(lineNumber, remainingChars + 1);
    }
    remainingChars -= line.length + 1;
    lineNumber++;
  }
  throw new Error('Algorithm failure');
}
function monacoPositionToOffset(expression, position) {
  const lines = expression.split(/\n/);
  return lines.slice(0, position.lineNumber).reduce((prev, current, index) => prev + (index === position.lineNumber - 1 ? position.column - 1 : current.length + 1), 0);
}
async function suggest({
  expression,
  zeroIndexedOffset,
  context,
  indexPattern,
  operationDefinitionMap,
  dataViews,
  unifiedSearch,
  dateHistogramInterval,
  dateRange
}) {
  const text = expression.substr(0, zeroIndexedOffset) + MARKER + expression.substr(zeroIndexedOffset);
  try {
    const ast = (0, _tinymath.parse)(text);
    const tokenInfo = getInfoAtZeroIndexedPosition(ast, zeroIndexedOffset);
    const tokenAst = tokenInfo === null || tokenInfo === void 0 ? void 0 : tokenInfo.ast;
    const isNamedArgument = (tokenInfo === null || tokenInfo === void 0 ? void 0 : tokenInfo.parent) && typeof tokenAst !== 'number' && tokenAst && 'type' in tokenAst && tokenAst.type === 'namedArgument';
    if (tokenInfo !== null && tokenInfo !== void 0 && tokenInfo.parent && (context.triggerCharacter === '=' || isNamedArgument)) {
      return await getNamedArgumentSuggestions({
        ast: tokenAst,
        unifiedSearch,
        dataViews,
        indexPattern,
        dateHistogramInterval,
        dateRange
      });
    } else if (tokenInfo !== null && tokenInfo !== void 0 && tokenInfo.parent) {
      return getArgumentSuggestions(tokenInfo.parent, tokenInfo.parent.args.findIndex(a => a === tokenAst), text, indexPattern, operationDefinitionMap);
    }
    if (typeof tokenAst === 'object' && Boolean(tokenAst.type === 'variable' || tokenAst.type === 'function')) {
      const nameWithMarker = tokenAst.type === 'function' ? tokenAst.name : tokenAst.value;
      return getFunctionSuggestions(nameWithMarker.split(MARKER)[0], indexPattern, operationDefinitionMap);
    }
  } catch (e) {
    // Fail silently
  }
  return {
    list: [],
    type: SUGGESTION_TYPE.FIELD
  };
}
function getPossibleFunctions(indexPattern, operationDefinitionMap) {
  const available = (0, _operations.memoizedGetAvailableOperationsByMetadata)(indexPattern, operationDefinitionMap);
  const possibleOperationNames = [];
  available.forEach(a => {
    if (a.operationMetaData.dataType === 'number' && !a.operationMetaData.isBucketed) {
      possibleOperationNames.push(...a.operations.filter(o => o.type !== 'managedReference' || o.usedInMath).map(o => o.operationType));
    }
  });
  return [...(0, _lodash.uniq)(possibleOperationNames), ...Object.keys(_util.tinymathFunctions)];
}
function getFunctionSuggestions(prefix, indexPattern, operationDefinitionMap) {
  return {
    list: (0, _lodash.uniq)(getPossibleFunctions(indexPattern, operationDefinitionMap).filter(func => (0, _lodash.startsWith)(func, prefix))).map(func => ({
      label: func,
      type: 'operation'
    })),
    type: SUGGESTION_TYPE.FUNCTIONS
  };
}
function getArgumentSuggestions(ast, position, expression, indexPattern, operationDefinitionMap) {
  const {
    name
  } = ast;
  const operation = operationDefinitionMap[name];
  if (!operation && !_util.tinymathFunctions[name]) {
    return {
      list: [],
      type: SUGGESTION_TYPE.FIELD
    };
  }
  const tinymathFunction = _util.tinymathFunctions[name];
  if (tinymathFunction) {
    if (tinymathFunction.positionalArguments[position]) {
      return {
        list: (0, _lodash.uniq)(getPossibleFunctions(indexPattern, operationDefinitionMap)).map(f => ({
          type: 'math',
          label: f
        })),
        type: SUGGESTION_TYPE.FUNCTIONS
      };
    }
    return {
      list: [],
      type: SUGGESTION_TYPE.FIELD
    };
  }
  if (position > 0 || !(0, _validation.hasFunctionFieldArgument)(operation.type)) {
    const {
      namedArguments
    } = (0, _util.groupArgsByType)(ast.args);
    const list = [];
    if (operation.filterable) {
      const hasFilterArgument = namedArguments.find(arg => arg.name === 'kql' || arg.name === 'lucene');
      if (!hasFilterArgument) {
        list.push('kql');
        list.push('lucene');
      }
    }
    if (operation.shiftable) {
      if (!namedArguments.find(arg => arg.name === 'shift')) {
        list.push('shift');
      }
    }
    if (operation.canReduceTimeRange) {
      if (!namedArguments.find(arg => arg.name === 'reducedTimeRange')) {
        list.push('reducedTimeRange');
      }
    }
    if ('operationParams' in operation) {
      // Exclude any previously used named args
      list.push(...operation.operationParams.filter(param =>
      // Keep the param if it's the first use
      !namedArguments.find(arg => arg.name === param.name)).map(p => p.name));
    }
    return {
      list,
      type: SUGGESTION_TYPE.NAMED_ARGUMENT
    };
  }
  if (operation.input === 'field' && position === 0) {
    const available = (0, _operations.memoizedGetAvailableOperationsByMetadata)(indexPattern, operationDefinitionMap);
    // TODO: This only allow numeric functions, will reject last_value(string) for example.
    const validOperation = available.filter(({
      operationMetaData
    }) => (operationMetaData.dataType === 'number' || operationMetaData.dataType === 'date') && !operationMetaData.isBucketed);
    if (validOperation.length) {
      const fields = validOperation.flatMap(op => op.operations).filter(op => op.operationType === operation.type).map(op => 'field' in op ? op.field : undefined).filter(_utils.nonNullable);
      const fieldArg = ast.args[0];
      const location = typeof fieldArg !== 'string' && fieldArg.location;
      let range;
      if (location) {
        const start = offsetToRowColumn(expression, location.min);
        // This accounts for any characters that the user has already typed
        const end = offsetToRowColumn(expression, location.max - MARKER.length);
        range = _monaco.monaco.Range.fromPositions(start, end);
      }
      return {
        list: fields,
        type: SUGGESTION_TYPE.FIELD,
        range
      };
    } else {
      return {
        list: [],
        type: SUGGESTION_TYPE.FIELD
      };
    }
  }
  if (operation.input === 'fullReference') {
    const available = (0, _operations.memoizedGetAvailableOperationsByMetadata)(indexPattern, operationDefinitionMap);
    const possibleOperationNames = [];
    available.forEach(a => {
      if (operation.requiredReferences.some(requirement => requirement.validateMetadata(a.operationMetaData))) {
        possibleOperationNames.push(...a.operations.filter(o => operation.requiredReferences.some(requirement => requirement.input.includes(o.type))).map(o => o.operationType));
      }
    });
    return {
      list: (0, _lodash.uniq)(possibleOperationNames).map(n => ({
        label: n,
        type: 'operation'
      })),
      type: SUGGESTION_TYPE.FUNCTIONS
    };
  }
  return {
    list: [],
    type: SUGGESTION_TYPE.FIELD
  };
}
const anchoredAbsoluteTimeShiftRegexp = /^(start|end)At\(/;
function computeAbsSuggestion(dateRange, prefix, value) {
  const refDate = prefix.startsWith('s') ? dateRange.fromDate : dateRange.toDate;
  return (0, _moment.default)(refDate).subtract((0, _common.parseTimeShift)(value), 'ms').toISOString();
}
async function getNamedArgumentSuggestions({
  ast,
  unifiedSearch,
  dataViews,
  indexPattern,
  dateHistogramInterval,
  dateRange
}) {
  if (ast.name === 'shift') {
    const validTimeShiftOptions = _time_shift_utils.timeShiftOptions.filter(({
      value
    }) => {
      if (dateHistogramInterval == null) return true;
      const parsedValue = (0, _common.parseTimeShift)(value);
      return parsedValue === 'previous' || parsedValue === 'invalid' || Number.isInteger(parsedValue.asMilliseconds() / dateHistogramInterval);
    }).map(({
      value
    }) => value);
    const absShift = ast.value.split(MARKER)[0];
    // Translate the relative time shifts into absolute ones
    if (anchoredAbsoluteTimeShiftRegexp.test(absShift)) {
      return {
        list: validTimeShiftOptions.map(value => `${computeAbsSuggestion(dateRange, absShift, value)})`),
        type: SUGGESTION_TYPE.SHIFTS
      };
    }
    const extraAbsSuggestions = ['startAt', 'endAt'].map(prefix => `${prefix}(${computeAbsSuggestion(dateRange, prefix, validTimeShiftOptions[0])})`);
    return {
      list: validTimeShiftOptions.concat(extraAbsSuggestions),
      type: SUGGESTION_TYPE.SHIFTS
    };
  }
  if (ast.name === 'reducedTimeRange') {
    return {
      list: _reduced_time_range_utils.reducedTimeRangeOptions.map(({
        value
      }) => value),
      type: SUGGESTION_TYPE.REDUCED_TIME_RANGES
    };
  }
  if (ast.name !== 'kql' && ast.name !== 'lucene') {
    return {
      list: [],
      type: SUGGESTION_TYPE.KQL
    };
  }
  if (!unifiedSearch.autocomplete.hasQuerySuggestions(ast.name === 'kql' ? 'kuery' : 'lucene')) {
    return {
      list: [],
      type: SUGGESTION_TYPE.KQL
    };
  }
  const query = ast.value.split(MARKER)[0];
  const position = ast.value.indexOf(MARKER) + 1;
  const suggestions = await unifiedSearch.autocomplete.getQuerySuggestions({
    language: ast.name === 'kql' ? 'kuery' : 'lucene',
    query,
    selectionStart: position,
    selectionEnd: position,
    indexPatterns: [await dataViews.get(indexPattern.id)],
    boolFilter: []
  });
  return {
    list: suggestions !== null && suggestions !== void 0 ? suggestions : [],
    type: SUGGESTION_TYPE.KQL
  };
}
const TRIGGER_SUGGESTION_COMMAND = {
  title: 'Trigger Suggestion Dialog',
  id: 'editor.action.triggerSuggest'
};
function getSuggestion(suggestion, type, operationDefinitionMap, triggerChar, range) {
  var _insertText;
  let kind = _monaco.monaco.languages.CompletionItemKind.Method;
  let label = typeof suggestion === 'string' ? suggestion : 'label' in suggestion ? suggestion.label : suggestion.text;
  let insertText;
  let insertTextRules;
  let detail = '';
  let command;
  let sortText = '';
  const filterText = label;
  switch (type) {
    case SUGGESTION_TYPE.SHIFTS:
      sortText = String(_time_shift_utils.timeShiftOptionOrder[label]).padStart(4, '0');
      break;
    case SUGGESTION_TYPE.REDUCED_TIME_RANGES:
      sortText = String(_reduced_time_range_utils.reducedTimeRangeOptionOrder[label]).padStart(4, '0');
      break;
    case SUGGESTION_TYPE.FIELD:
      kind = _monaco.monaco.languages.CompletionItemKind.Value;
      // Look for unsafe characters
      if (_util.unquotedStringRegex.test(label)) {
        insertText = `'${label.replaceAll(`'`, "\\'")}'`;
      }
      break;
    case SUGGESTION_TYPE.FUNCTIONS:
      insertText = `${label}($0)`;
      insertTextRules = _monaco.monaco.languages.CompletionItemInsertTextRule.InsertAsSnippet;
      if (typeof suggestion !== 'string') {
        if ('text' in suggestion) break;
        label = (0, _formula_help.getFunctionSignatureLabel)(suggestion.label, operationDefinitionMap);
        const tinymathFunction = _util.tinymathFunctions[suggestion.label];
        if (tinymathFunction) {
          detail = 'TinyMath';
          kind = _monaco.monaco.languages.CompletionItemKind.Method;
        } else {
          kind = _monaco.monaco.languages.CompletionItemKind.Constant;
          detail = 'Elasticsearch';
          // Always put ES functions first
          sortText = `0${label}`;
          command = TRIGGER_SUGGESTION_COMMAND;
        }
      }
      break;
    case SUGGESTION_TYPE.NAMED_ARGUMENT:
      kind = _monaco.monaco.languages.CompletionItemKind.Keyword;
      if (label === 'kql' || label === 'lucene' || label === 'shift' || label === 'reducedTimeRange') {
        command = TRIGGER_SUGGESTION_COMMAND;
        insertText = `${label}='$0'`;
        insertTextRules = _monaco.monaco.languages.CompletionItemInsertTextRule.InsertAsSnippet;
        sortText = `zzz${label}`;
      }
      label = `${label}=`;
      detail = '';
      break;
    case SUGGESTION_TYPE.KQL:
      if (triggerChar === ':') {
        insertText = `${triggerChar} ${label}`;
      } else {
        // concatenate KQL suggestion for faster query composition
        command = TRIGGER_SUGGESTION_COMMAND;
      }
      if (label.includes(`'`)) {
        insertText = (insertText || label).replaceAll(`'`, "\\'");
      }
      break;
  }
  return {
    detail,
    kind,
    label,
    insertText: (_insertText = insertText) !== null && _insertText !== void 0 ? _insertText : label,
    insertTextRules,
    command,
    additionalTextEdits: [],
    // @ts-expect-error Monaco says this type is required, but provides a default value
    range,
    sortText,
    filterText
  };
}
function getOperationTypeHelp(name, operationDefinitionMap) {
  const {
    description: descriptionInMarkdown,
    examples
  } = (0, _formula_help.getHelpTextContent)(name, operationDefinitionMap);
  const examplesInMarkdown = examples.length ? `\n\n**${_i18n.i18n.translate('xpack.lens.formulaExampleMarkdown', {
    defaultMessage: 'Examples'
  })}**

  ${examples.map(example => `\`${example}\``).join('\n\n')}` : '';
  return {
    value: `${descriptionInMarkdown}${examplesInMarkdown}`
  };
}
function getSignaturesForFunction(name, operationDefinitionMap) {
  if (_util.tinymathFunctions[name]) {
    const stringify = (0, _formula_help.getFunctionSignatureLabel)(name, operationDefinitionMap);
    const documentation = _util.tinymathFunctions[name].help.replace(/\n/g, '\n\n');
    return [{
      label: stringify,
      documentation: {
        value: documentation
      },
      parameters: _util.tinymathFunctions[name].positionalArguments.map(arg => ({
        label: arg.name,
        documentation: arg.optional ? _i18n.i18n.translate('xpack.lens.formula.optionalArgument', {
          defaultMessage: 'Optional. Default value is {defaultValue}',
          values: {
            defaultValue: arg.defaultValue
          }
        }) : ''
      }))
    }];
  }
  if (operationDefinitionMap[name]) {
    const def = operationDefinitionMap[name];
    const firstParam = (0, _validation.hasFunctionFieldArgument)(name) ? {
      label: def.input === 'field' ? 'field' : def.input === 'fullReference' ? 'function' : ''
    } : null;
    const functionLabel = (0, _formula_help.getFunctionSignatureLabel)(name, operationDefinitionMap);
    const documentation = getOperationTypeHelp(name, operationDefinitionMap);
    if ('operationParams' in def && def.operationParams) {
      return [{
        label: functionLabel,
        parameters: [...(firstParam ? [firstParam] : []), ...def.operationParams.map(arg => ({
          label: `${arg.name}=${arg.type}`,
          documentation: arg.required ? _i18n.i18n.translate('xpack.lens.formula.requiredArgument', {
            defaultMessage: 'Required'
          }) : ''
        }))],
        documentation
      }];
    }
    return [{
      label: functionLabel,
      parameters: firstParam ? [firstParam] : [],
      documentation
    }];
  }
  return [];
}
function getSignatureHelp(expression, position, operationDefinitionMap) {
  const text = expression.substr(0, position) + MARKER + expression.substr(position);
  try {
    const ast = (0, _tinymath.parse)(text);
    const tokenInfo = getInfoAtZeroIndexedPosition(ast, position);
    let signatures = [];
    let index = 0;
    if (tokenInfo !== null && tokenInfo !== void 0 && tokenInfo.parent) {
      const name = tokenInfo.parent.name;
      // reference equality is fine here because of the way the getInfo function works
      index = tokenInfo.parent.args.findIndex(arg => arg === tokenInfo.ast);
      signatures = getSignaturesForFunction(name, operationDefinitionMap);
    } else if (typeof (tokenInfo === null || tokenInfo === void 0 ? void 0 : tokenInfo.ast) === 'object' && tokenInfo.ast.type === 'function') {
      const name = tokenInfo.ast.name;
      signatures = getSignaturesForFunction(name, operationDefinitionMap);
    }
    if (signatures.length) {
      return {
        value: {
          // remove the documentation
          signatures: signatures.map(({
            documentation,
            ...signature
          }) => ({
            ...signature,
            // extract only the first section (usually few lines)
            documentation: {
              value: documentation.value.split('\n\n')[0]
            }
          })),
          activeParameter: index,
          activeSignature: 0
        },
        dispose: () => {}
      };
    }
  } catch (e) {
    // do nothing
  }
  return {
    value: {
      signatures: [],
      activeParameter: 0,
      activeSignature: 0
    },
    dispose: () => {}
  };
}
function getHover(expression, position, operationDefinitionMap) {
  try {
    const ast = (0, _tinymath.parse)(expression);
    const tokenInfo = getInfoAtZeroIndexedPosition(ast, position);
    if (!tokenInfo || typeof tokenInfo.ast === 'number' || !('name' in tokenInfo.ast)) {
      return {
        contents: []
      };
    }
    const name = tokenInfo.ast.name;
    const signatures = getSignaturesForFunction(name, operationDefinitionMap);
    if (signatures.length) {
      const {
        label
      } = signatures[0];
      return {
        contents: [{
          value: label
        }]
      };
    }
  } catch (e) {
    // do nothing
  }
  return {
    contents: []
  };
}
function getTokenInfo(expression, position) {
  const text = expression.substr(0, position) + MARKER + expression.substr(position);
  try {
    const ast = (0, _tinymath.parse)(text);
    return getInfoAtZeroIndexedPosition(ast, position);
  } catch (e) {
    return;
  }
}