"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.makeRouterWithFleetAuthz = makeRouterWithFleetAuthz;
var _request_store = require("../request_store");
var _security = require("./security");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function makeRouterWithFleetAuthz(router, logger) {
  const routerAuthzWrapper = async ({
    context,
    request,
    response,
    handler,
    hasRequiredAuthz
  }) => {
    if (!(0, _security.checkSecurityEnabled)()) {
      const securityEnabledInfo = 'Kibana security must be enabled to use Fleet';
      logger.info(securityEnabledInfo);
      return response.forbidden({
        body: {
          message: securityEnabledInfo
        }
      });
    }
    const requestedAuthz = await (0, _security.getAuthzFromRequest)(request);
    if ((0, _security.doesNotHaveRequiredFleetAuthz)(requestedAuthz, hasRequiredAuthz)) {
      logger.info(`User does not have required fleet authz to access path: ${request.route.path}`);
      return response.forbidden();
    }
    return handler(context, request, response);
  };
  const requestContextWrapper = async ({
    context,
    request,
    response,
    handler
  }) => {
    return (0, _request_store.getRequestStore)().run(request, () => handler(context, request, response));
  };
  const fleetHandlerWrapper = async ({
    context,
    request,
    response,
    handler,
    hasRequiredAuthz
  }) => {
    return requestContextWrapper({
      context,
      request,
      response,
      handler: (handlerContext, handlerRequest, handlerResponse) => routerAuthzWrapper({
        context: handlerContext,
        request: handlerRequest,
        response: handlerResponse,
        handler,
        hasRequiredAuthz
      })
    });
  };
  const fleetAuthzRouter = {
    get: ({
      fleetAuthz: hasRequiredAuthz,
      ...options
    }, handler) => {
      router.get(options, (context, request, response) => fleetHandlerWrapper({
        context,
        request,
        response,
        handler,
        hasRequiredAuthz
      }));
    },
    delete: ({
      fleetAuthz: hasRequiredAuthz,
      ...options
    }, handler) => {
      router.delete(options, (context, request, response) => fleetHandlerWrapper({
        context,
        request,
        response,
        handler,
        hasRequiredAuthz
      }));
    },
    post: ({
      fleetAuthz: hasRequiredAuthz,
      ...options
    }, handler) => {
      router.post(options, (context, request, response) => fleetHandlerWrapper({
        context,
        request,
        response,
        handler,
        hasRequiredAuthz
      }));
    },
    put: ({
      fleetAuthz: hasRequiredAuthz,
      ...options
    }, handler) => {
      router.put(options, (context, request, response) => fleetHandlerWrapper({
        context,
        request,
        response,
        handler,
        hasRequiredAuthz
      }));
    },
    patch: ({
      fleetAuthz: hasRequiredAuthz,
      ...options
    }, handler) => {
      router.patch(options, (context, request, response) => fleetHandlerWrapper({
        context,
        request,
        response,
        handler,
        hasRequiredAuthz
      }));
    },
    handleLegacyErrors: handler => router.handleLegacyErrors(handler),
    getRoutes: () => router.getRoutes(),
    routerPath: router.routerPath,
    versioned: router.versioned
  };
  return fleetAuthzRouter;
}