"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSearchApplicationsRoutes = registerSearchApplicationsRoutes;
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _error_codes = require("../../../common/types/error_codes");
var _create_api_key = require("../../lib/search_applications/create_api_key");
var _fetch_indices_stats = require("../../lib/search_applications/fetch_indices_stats");
var _field_capabilities = require("../../lib/search_applications/field_capabilities");
var _create_error = require("../../utils/create_error");
var _elasticsearch_error_handler = require("../../utils/elasticsearch_error_handler");
var _identify_exceptions = require("../../utils/identify_exceptions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerSearchApplicationsRoutes({
  log,
  router
}) {
  router.get({
    path: '/internal/enterprise_search/search_applications',
    validate: {
      query: _configSchema.schema.object({
        from: _configSchema.schema.number({
          defaultValue: 0,
          min: 0
        }),
        q: _configSchema.schema.maybe(_configSchema.schema.string()),
        size: _configSchema.schema.number({
          defaultValue: 10,
          min: 1
        })
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const engines = await client.asCurrentUser.searchApplication.list(request.query);
    return response.ok({
      body: engines
    });
  }));
  router.get({
    path: '/internal/enterprise_search/search_applications/{engine_name}',
    validate: {
      params: _configSchema.schema.object({
        engine_name: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const engine = await client.asCurrentUser.searchApplication.get({
      name: request.params.engine_name
    });
    const indicesStats = await (0, _fetch_indices_stats.fetchIndicesStats)(client, engine.indices);
    return response.ok({
      body: {
        ...engine,
        indices: indicesStats
      }
    });
  }));
  router.put({
    path: '/internal/enterprise_search/search_applications/{engine_name}',
    validate: {
      body: _configSchema.schema.object({
        indices: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        name: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      params: _configSchema.schema.object({
        engine_name: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        create: _configSchema.schema.maybe(_configSchema.schema.boolean())
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    var _error$meta, _error$meta$body, _error$meta$body$erro, _error$meta$body$erro2;
    const {
      client
    } = (await context.core).elasticsearch;
    try {
      const engine = await client.asCurrentUser.searchApplication.put({
        ...request.query,
        name: request.params.engine_name,
        search_application: {
          indices: request.body.indices,
          name: request.params.engine_name,
          updated_at_millis: Date.now()
        }
      });
      return response.ok({
        body: engine
      });
    } catch (error) {
      switch (true) {
        case (0, _identify_exceptions.isVersionConflictEngineException)(error):
          return (0, _create_error.createError)({
            errorCode: _error_codes.ErrorCode.SEARCH_APPLICATION_ALREADY_EXISTS,
            message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.createSearchApplication.searchApplciationExistsError', {
              defaultMessage: 'Search application name already taken. Choose another name.'
            }),
            response,
            statusCode: 409
          });
        case (0, _identify_exceptions.isInvalidSearchApplicationNameException)(error):
          let exceptionReason = '';
          const unSupportedCharacters = (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$body = _error$meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : (_error$meta$body$erro2 = _error$meta$body$erro.reason) === null || _error$meta$body$erro2 === void 0 ? void 0 : _error$meta$body$erro2.match(/\[(.*?)\]|'(.*?)'/gi);
          if (unSupportedCharacters && unSupportedCharacters.length === 2) {
            exceptionReason = 'Search application name must not contain: ' + unSupportedCharacters[1].replace(/\'(.*?)\'/g, ' $& ');
          }
          return (0, _create_error.createError)({
            errorCode: _error_codes.ErrorCode.SEARCH_APPLICATION_NAME_INVALID,
            message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.createSearchApplication.searchApplicationInvalidName', {
              defaultMessage: 'Invalid Search application name. {exceptionReason}',
              values: {
                exceptionReason
              }
            }),
            response,
            statusCode: 400
          });
        default:
          throw error;
      }
    }
  }));
  router.delete({
    path: '/internal/enterprise_search/search_applications/{engine_name}',
    validate: {
      params: _configSchema.schema.object({
        engine_name: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const engine = await client.asCurrentUser.searchApplication.delete({
      name: request.params.engine_name
    });
    return response.ok({
      body: engine
    });
  }));
  router.post({
    path: '/internal/enterprise_search/search_applications/{engine_name}/search',
    validate: {
      body: _configSchema.schema.object({}, {
        unknowns: 'allow'
      }),
      params: _configSchema.schema.object({
        engine_name: _configSchema.schema.string(),
        from: _configSchema.schema.maybe(_configSchema.schema.number()),
        size: _configSchema.schema.maybe(_configSchema.schema.number())
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const engines = await client.asCurrentUser.search({
      ...request.body,
      index: request.params.engine_name
    });
    return response.ok({
      body: engines
    });
  }));
  router.post({
    path: '/internal/enterprise_search/search_applications/{engine_name}/api_key',
    validate: {
      body: _configSchema.schema.object({
        keyName: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        engine_name: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const engineName = decodeURIComponent(request.params.engine_name);
    const {
      keyName
    } = request.body;
    const {
      client
    } = (await context.core).elasticsearch;
    const apiKey = await (0, _create_api_key.createApiKey)(client, engineName, keyName);
    return response.ok({
      body: {
        apiKey
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.get({
    path: '/internal/enterprise_search/search_applications/{engine_name}/field_capabilities',
    validate: {
      params: _configSchema.schema.object({
        engine_name: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    try {
      const {
        client
      } = (await context.core).elasticsearch;
      const engine = await client.asCurrentUser.searchApplication.get({
        name: request.params.engine_name
      });
      const data = await (0, _field_capabilities.fetchSearchApplicationFieldCapabilities)(client, engine);
      return response.ok({
        body: data,
        headers: {
          'content-type': 'application/json'
        }
      });
    } catch (e) {
      if ((0, _identify_exceptions.isNotFoundException)(e)) {
        return (0, _create_error.createError)({
          errorCode: _error_codes.ErrorCode.ENGINE_NOT_FOUND,
          message: 'Could not find engine',
          response,
          statusCode: 404
        });
      }
      throw e;
    }
  }));
}