"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertToResultFormat = exports.convertIdToMeta = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * The `promoted` and `hidden` keys from the internal curations endpoints
 * currently return a document data structure that our Result component can't
 * correctly parse - we need to attempt to naively transform the data in order
 * to display it in a Result
 *
 * TODO: Ideally someday we can update our internal curations endpoint to return
 * the same Result-ready data structure that the `organic` endpoint uses, and
 * remove this file when that happens
 */

const mergeMetas = (partialMeta, secondPartialMeta) => {
  return {
    ...(partialMeta || {}),
    ...secondPartialMeta
  };
};
const isNestedObject = value => {
  if (Array.isArray(value)) {
    return value.reduce((isNested, currentValue) => isNested || isNestedObject(currentValue), false);
  }
  return value === null || typeof value === 'object';
};
const convertToResultFormat = document => {
  // Convert `key: 'value'` into `key: { raw: 'value' }`
  const result = Object.entries(document).reduce((acc, [key, value]) => {
    acc[key] = isNestedObject(value) || Object.prototype.hasOwnProperty.call(value, 'raw') ? value : {
      raw: value
    };
    return acc;
  }, {});
  result._meta = mergeMetas(result._meta, convertIdToMeta(document.id));
  return result;
};
exports.convertToResultFormat = convertToResultFormat;
const convertIdToMeta = id => {
  const splitId = id.split('|');
  const isMetaEngine = splitId.length > 1;
  return isMetaEngine ? {
    engine: splitId[0],
    id: splitId[1]
  } : {
    id
  };
  // Note: We're casting this as _meta even though `engine` is missing,
  // since for source engines the engine shouldn't matter / be displayed,
  // but if needed we could likely populate this from EngineLogic.values
};
exports.convertIdToMeta = convertIdToMeta;