"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.copyValueToClipboard = exports.copyColumnValuesToClipboard = exports.copyColumnNameToClipboard = void 0;
var _eui = require("@elastic/eui");
var _i18n = require("@kbn/i18n");
var _convert_value_to_string = require("./convert_value_to_string");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const WARNING_FOR_FORMULAS = _i18n.i18n.translate('discover.grid.copyEscapedValueWithFormulasToClipboardWarningText', {
  defaultMessage: 'Values may contain formulas that are escaped.'
});
const COPY_FAILED_ERROR_MESSAGE = _i18n.i18n.translate('discover.grid.copyFailedErrorText', {
  defaultMessage: 'Unable to copy to clipboard in this browser'
});
const copyValueToClipboard = ({
  rowIndex,
  columnId,
  toastNotifications,
  valueToStringConverter
}) => {
  const result = valueToStringConverter(rowIndex, columnId);
  const valueFormatted = result.formattedString;
  const copied = (0, _eui.copyToClipboard)(valueFormatted);
  if (!copied) {
    toastNotifications.addWarning({
      title: COPY_FAILED_ERROR_MESSAGE
    });
    return null;
  }
  const toastTitle = _i18n.i18n.translate('discover.grid.copyValueToClipboard.toastTitle', {
    defaultMessage: 'Copied to clipboard'
  });
  if (result.withFormula) {
    toastNotifications.addWarning({
      title: toastTitle,
      text: WARNING_FOR_FORMULAS
    });
  } else {
    toastNotifications.addInfo({
      title: toastTitle
    });
  }
  return valueFormatted;
};
exports.copyValueToClipboard = copyValueToClipboard;
const copyColumnValuesToClipboard = async ({
  columnId,
  columnDisplayName,
  toastNotifications,
  valueToStringConverter,
  rowsCount
}) => {
  const nameFormattedResult = (0, _convert_value_to_string.convertNameToString)(columnDisplayName);
  let withFormula = nameFormattedResult.withFormula;
  const valuesFormatted = [...Array(rowsCount)].map((_, rowIndex) => {
    const result = valueToStringConverter(rowIndex, columnId, {
      compatibleWithCSV: true
    });
    withFormula = withFormula || result.withFormula;
    return result.formattedString;
  });
  const textToCopy = `${nameFormattedResult.formattedString}\n${valuesFormatted.join('\n')}`;
  let copied;
  try {
    var _window$navigator, _window$navigator$cli;
    // try to copy without browser styles
    await ((_window$navigator = window.navigator) === null || _window$navigator === void 0 ? void 0 : (_window$navigator$cli = _window$navigator.clipboard) === null || _window$navigator$cli === void 0 ? void 0 : _window$navigator$cli.writeText(textToCopy));
    copied = true;
  } catch (error) {
    copied = (0, _eui.copyToClipboard)(textToCopy);
  }
  if (!copied) {
    toastNotifications.addWarning({
      title: COPY_FAILED_ERROR_MESSAGE
    });
    return null;
  }
  const toastTitle = _i18n.i18n.translate('discover.grid.copyColumnValuesToClipboard.toastTitle', {
    defaultMessage: 'Values of "{column}" column copied to clipboard',
    values: {
      column: columnDisplayName
    }
  });
  if (withFormula) {
    toastNotifications.addWarning({
      title: toastTitle,
      text: WARNING_FOR_FORMULAS
    });
  } else {
    toastNotifications.addInfo({
      title: toastTitle
    });
  }
  return textToCopy;
};
exports.copyColumnValuesToClipboard = copyColumnValuesToClipboard;
const copyColumnNameToClipboard = ({
  columnDisplayName,
  toastNotifications
}) => {
  const nameFormattedResult = (0, _convert_value_to_string.convertNameToString)(columnDisplayName);
  const textToCopy = nameFormattedResult.formattedString;
  const copied = (0, _eui.copyToClipboard)(textToCopy);
  if (!copied) {
    toastNotifications.addWarning({
      title: COPY_FAILED_ERROR_MESSAGE
    });
    return null;
  }
  const toastTitle = _i18n.i18n.translate('discover.grid.copyColumnNameToClipboard.toastTitle', {
    defaultMessage: 'Copied to clipboard'
  });
  if (nameFormattedResult.withFormula) {
    toastNotifications.addWarning({
      title: toastTitle,
      text: WARNING_FOR_FORMULAS
    });
  } else {
    toastNotifications.addInfo({
      title: toastTitle
    });
  }
  return textToCopy;
};
exports.copyColumnNameToClipboard = copyColumnNameToClipboard;