"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.copySavedSearch = copySavedSearch;
exports.getDefaultAppState = getDefaultAppState;
exports.getSavedSearchContainer = getSavedSearchContainer;
exports.isEqualSavedSearch = isEqualSavedSearch;
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _restore_from_saved_search = require("../../../services/saved_searches/restore_from_saved_search");
var _update_saved_search = require("../utils/update_saved_search");
var _add_log = require("../../../utils/add_log");
var _state_helpers = require("../../../utils/state_helpers");
var _get_state_defaults = require("../utils/get_state_defaults");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function getSavedSearchContainer({
  services,
  globalStateContainer
}) {
  const initialSavedSearch = services.savedSearch.getNew();
  const savedSearchInitial$ = new _rxjs.BehaviorSubject(initialSavedSearch);
  const savedSearchCurrent$ = new _rxjs.BehaviorSubject(copySavedSearch(initialSavedSearch));
  const hasChanged$ = new _rxjs.BehaviorSubject(false);
  const set = savedSearch => {
    (0, _add_log.addLog)('[savedSearch] set', savedSearch);
    hasChanged$.next(false);
    savedSearchCurrent$.next(savedSearch);
    savedSearchInitial$.next(copySavedSearch(savedSearch));
    return savedSearch;
  };
  const getState = () => savedSearchCurrent$.getValue();
  const getInitial$ = () => savedSearchInitial$;
  const getCurrent$ = () => savedSearchCurrent$;
  const getHasChanged$ = () => hasChanged$;
  const getTitle = () => savedSearchCurrent$.getValue().title;
  const getId = () => savedSearchCurrent$.getValue().id;
  const newSavedSearch = async nextDataView => {
    (0, _add_log.addLog)('[savedSearch] new', {
      nextDataView
    });
    const dataView = nextDataView !== null && nextDataView !== void 0 ? nextDataView : getState().searchSource.getField('index');
    const nextSavedSearch = services.savedSearch.getNew();
    nextSavedSearch.searchSource.setField('index', dataView);
    const newAppState = getDefaultAppState(nextSavedSearch, services);
    const nextSavedSearchToSet = (0, _update_saved_search.updateSavedSearch)({
      savedSearch: {
        ...nextSavedSearch
      },
      dataView,
      state: newAppState,
      globalStateContainer,
      services
    });
    return set(nextSavedSearchToSet);
  };
  const persist = async (nextSavedSearch, saveOptions) => {
    (0, _add_log.addLog)('[savedSearch] persist', {
      nextSavedSearch,
      saveOptions
    });
    (0, _update_saved_search.updateSavedSearch)({
      savedSearch: nextSavedSearch,
      globalStateContainer,
      services,
      useFilterAndQueryServices: true
    });
    const id = await services.savedSearch.save(nextSavedSearch, saveOptions || {});
    if (id) {
      set(nextSavedSearch);
    }
    return {
      id
    };
  };
  const update = ({
    nextDataView,
    nextState,
    useFilterAndQueryServices
  }) => {
    (0, _add_log.addLog)('[savedSearch] update', {
      nextDataView,
      nextState
    });
    const previousSavedSearch = getState();
    const dataView = nextDataView ? nextDataView : previousSavedSearch.searchSource.getField('index');
    const nextSavedSearch = (0, _update_saved_search.updateSavedSearch)({
      savedSearch: {
        ...previousSavedSearch
      },
      dataView,
      state: nextState || {},
      globalStateContainer,
      services,
      useFilterAndQueryServices
    });
    const hasChanged = !isEqualSavedSearch(savedSearchInitial$.getValue(), nextSavedSearch);
    hasChanged$.next(hasChanged);
    savedSearchCurrent$.next(nextSavedSearch);
    (0, _add_log.addLog)('[savedSearch] update done', nextSavedSearch);
    return nextSavedSearch;
  };
  const load = async (id, dataView) => {
    (0, _add_log.addLog)('[savedSearch] load', {
      id,
      dataView
    });
    const loadedSavedSearch = await services.savedSearch.get(id);
    if (!loadedSavedSearch.searchSource.getField('index') && dataView) {
      loadedSavedSearch.searchSource.setField('index', dataView);
    }
    (0, _restore_from_saved_search.restoreStateFromSavedSearch)({
      savedSearch: loadedSavedSearch,
      timefilter: services.timefilter
    });
    return set(loadedSavedSearch);
  };
  return {
    getCurrent$,
    getHasChanged$,
    getId,
    getInitial$,
    getState,
    getTitle,
    load,
    new: newSavedSearch,
    persist,
    set,
    update
  };
}

/**
 * Copies a saved search object, due to the stateful nature of searchSource it has to be copied with a dedicated function
 * @param savedSearch
 */
function copySavedSearch(savedSearch) {
  return {
    ...savedSearch,
    ...{
      searchSource: savedSearch.searchSource.createCopy()
    }
  };
}
function getDefaultAppState(savedSearch, services) {
  return (0, _state_helpers.handleSourceColumnState)((0, _get_state_defaults.getStateDefaults)({
    savedSearch,
    services
  }), services.uiSettings);
}
function isEqualSavedSearch(savedSearchPrev, savedSearchNext) {
  const {
    searchSource: prevSearchSource,
    ...prevSavedSearch
  } = savedSearchPrev;
  const {
    searchSource: nextSearchSource,
    ...nextSavedSearchWithoutSearchSource
  } = savedSearchNext;
  const keys = new Set([...Object.keys(prevSavedSearch), ...Object.keys(nextSavedSearchWithoutSearchSource)]);
  const savedSearchDiff = [...keys].filter(key => {
    // @ts-expect-error
    return !(0, _lodash.isEqual)(prevSavedSearch[key], nextSavedSearchWithoutSearchSource[key]);
  });
  const searchSourceDiff = !(0, _lodash.isEqual)(prevSearchSource.getField('filter'), nextSearchSource.getField('filter')) || !(0, _lodash.isEqual)(prevSearchSource.getField('query'), nextSearchSource.getField('query')) || !(0, _lodash.isEqual)(prevSearchSource.getField('index'), nextSearchSource.getField('index'));
  const hasChanged = Boolean(savedSearchDiff.length || searchSourceDiff);
  if (hasChanged) {
    (0, _add_log.addLog)('[savedSearch] difference between initial and changed version', searchSourceDiff);
  }
  return !hasChanged;
}