"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useTextBasedQueryLanguage = useTextBasedQueryLanguage;
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _react = require("react");
var _public = require("@kbn/saved-search-plugin/public");
var _discover_state_provider = require("../services/discover_state_provider");
var _get_valid_view_mode = require("../utils/get_valid_view_mode");
var _types = require("../../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const MAX_NUM_OF_COLUMNS = 50;

/**
 * Hook to take care of text based query language state transformations when a new result is returned
 * If necessary this is setting displayed columns and selected data view
 */
function useTextBasedQueryLanguage({
  dataViews,
  stateContainer
}) {
  const prev = (0, _react.useRef)({
    columns: [],
    query: undefined
  });
  const indexTitle = (0, _react.useRef)('');
  const savedSearch = (0, _discover_state_provider.useSavedSearchInitial)();
  const cleanup = (0, _react.useCallback)(() => {
    if (prev.current.query) {
      // cleanup when it's not a text based query lang
      prev.current = {
        columns: [],
        query: undefined
      };
      indexTitle.current = '';
    }
  }, []);
  (0, _react.useEffect)(() => {
    const subscription = stateContainer.dataState.data$.documents$.subscribe(async next => {
      var _next$result;
      const {
        query,
        recordRawType
      } = next;
      if (!query || next.fetchStatus === _types.FetchStatus.ERROR) {
        return;
      }
      const sendComplete = () => {
        stateContainer.dataState.data$.documents$.next({
          ...next,
          fetchStatus: _types.FetchStatus.COMPLETE
        });
      };
      const {
        columns: stateColumns,
        index,
        viewMode
      } = stateContainer.appState.getState();
      let nextColumns = [];
      const isTextBasedQueryLang = recordRawType === 'plain' && (0, _esQuery.isOfAggregateQueryType)(query) && 'sql' in query;
      const hasResults = Boolean((_next$result = next.result) === null || _next$result === void 0 ? void 0 : _next$result.length);
      const initialFetch = !prev.current.columns.length;
      if (isTextBasedQueryLang) {
        if (next.fetchStatus !== _types.FetchStatus.PARTIAL) {
          return;
        }
        if (hasResults) {
          // check if state needs to contain column transformation due to a different columns in the resultset
          const firstRow = next.result[0];
          const firstRowColumns = Object.keys(firstRow.raw).slice(0, MAX_NUM_OF_COLUMNS);
          if (!(0, _lodash.isEqual)(firstRowColumns, prev.current.columns) && !(0, _lodash.isEqual)(query, prev.current.query)) {
            prev.current = {
              columns: firstRowColumns,
              query
            };
            nextColumns = firstRowColumns;
          }
          if (firstRowColumns && initialFetch) {
            prev.current = {
              columns: firstRowColumns,
              query
            };
          }
        }
        const indexPatternFromQuery = (0, _esQuery.getIndexPatternFromSQLQuery)(query.sql);
        const dataViewObj = stateContainer.internalState.getState().dataView;

        // don't set the columns on initial fetch, to prevent overwriting existing state
        const addColumnsToState = Boolean(nextColumns.length && (!initialFetch || !(stateColumns !== null && stateColumns !== void 0 && stateColumns.length)));
        // no need to reset index to state if it hasn't changed
        const addDataViewToState = Boolean((dataViewObj === null || dataViewObj === void 0 ? void 0 : dataViewObj.id) !== index) || initialFetch;
        const queryChanged = indexPatternFromQuery !== indexTitle.current;
        if (!addColumnsToState && !queryChanged) {
          sendComplete();
          return;
        }
        if (queryChanged) {
          indexTitle.current = indexPatternFromQuery;
        }
        const nextState = {
          ...(addDataViewToState && {
            index: dataViewObj.id
          }),
          ...(addColumnsToState && {
            columns: nextColumns
          }),
          ...(viewMode === _public.VIEW_MODE.AGGREGATED_LEVEL && {
            viewMode: (0, _get_valid_view_mode.getValidViewMode)({
              viewMode,
              isTextBasedQueryMode: true
            })
          })
        };
        await stateContainer.appState.replaceUrlState(nextState);
        sendComplete();
      } else {
        // cleanup for a "regular" query
        cleanup();
      }
    });
    return () => {
      // cleanup for e.g. when savedSearch is switched
      cleanup();
      subscription.unsubscribe();
    };
  }, [dataViews, stateContainer, savedSearch, cleanup]);
}