"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.syncQueryStateWithUrl = exports.syncGlobalQueryStateWithUrl = void 0;
var _public = require("@kbn/kibana-utils-plugin/public");
var _esQuery = require("@kbn/es-query");
var _connect_to_query_state = require("./connect_to_query_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const GLOBAL_STATE_STORAGE_KEY = '_g';

/**
 * Helper to sync global query state {@link GlobalQueryStateFromUrl} with the URL (`_g` query param that is preserved between apps)
 * @param QueryService: either setup or start
 * @param kbnUrlStateStorage to use for syncing
 */
const syncGlobalQueryStateWithUrl = (query, kbnUrlStateStorage) => {
  const {
    timefilter: {
      timefilter
    },
    filterManager
  } = query;
  const defaultState = {
    time: timefilter.getTime(),
    refreshInterval: timefilter.getRefreshInterval(),
    filters: filterManager.getGlobalFilters()
  };

  // retrieve current state from `_g` url
  const initialStateFromUrl = kbnUrlStateStorage.get(GLOBAL_STATE_STORAGE_KEY);

  // remember whether there was info in the URL
  const hasInheritedQueryFromUrl = Boolean(initialStateFromUrl && Object.keys(initialStateFromUrl).length);

  // prepare initial state, whatever was in URL takes precedences over current state in services
  const initialState = {
    ...defaultState,
    ...initialStateFromUrl
  };
  const globalQueryStateContainer = (0, _public.createStateContainer)(initialState);
  const stopSyncingWithStateContainer = (0, _connect_to_query_state.connectToQueryState)(query, globalQueryStateContainer, {
    refreshInterval: true,
    time: true,
    filters: _esQuery.FilterStateStore.GLOBAL_STATE
  });

  // if there weren't any initial state in url,
  // then put _g key into url
  if (!initialStateFromUrl) {
    kbnUrlStateStorage.set(GLOBAL_STATE_STORAGE_KEY, initialState, {
      replace: true
    });
  }

  // trigger initial syncing from state container to services if needed
  globalQueryStateContainer.set(initialState);
  const {
    start,
    stop: stopSyncingWithUrl
  } = (0, _public.syncState)({
    stateStorage: kbnUrlStateStorage,
    stateContainer: {
      ...globalQueryStateContainer,
      set: state => {
        if (state) {
          // syncState utils requires to handle incoming "null" value
          globalQueryStateContainer.set(state);
        }
      }
    },
    storageKey: GLOBAL_STATE_STORAGE_KEY
  });
  start();
  return {
    stop: () => {
      stopSyncingWithStateContainer();
      stopSyncingWithUrl();
    },
    hasInheritedQueryFromUrl
  };
};

/**
 * @deprecated use {@link syncGlobalQueryStateWithUrl} instead
 */
exports.syncGlobalQueryStateWithUrl = syncGlobalQueryStateWithUrl;
const syncQueryStateWithUrl = syncGlobalQueryStateWithUrl;
exports.syncQueryStateWithUrl = syncQueryStateWithUrl;