"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.startSyncingDashboardUrlState = exports.loadAndRemoveDashboardState = exports.isPanelVersionTooOld = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _operators = require("rxjs/operators");
var _satisfies = _interopRequireDefault(require("semver/functions/satisfies"));
var _common = require("@kbn/kibana-utils-plugin/common");
var _common2 = require("../../../common");
var _plugin_services = require("../../services/plugin_services");
var _dashboard_app_strings = require("../_dashboard_app_strings");
var _dashboard_constants = require("../../dashboard_constants");
var _load_dashboard_state = require("../../services/dashboard_content_management/lib/load_dashboard_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * We no longer support loading panels from a version older than 7.3 in the URL.
 * @returns whether or not there is a panel in the URL state saved with a version before 7.3
 */
const isPanelVersionTooOld = panels => {
  for (const panel of panels) {
    if (!panel.version || (0, _satisfies.default)(panel.version, '<7.3')) return true;
  }
  return false;
};
exports.isPanelVersionTooOld = isPanelVersionTooOld;
function getPanelsMap(appStateInUrl) {
  if (!appStateInUrl.panels) {
    return undefined;
  }
  if (appStateInUrl.panels.length === 0) {
    return {};
  }
  if (isPanelVersionTooOld(appStateInUrl.panels)) {
    _plugin_services.pluginServices.getServices().notifications.toasts.addWarning((0, _dashboard_app_strings.getPanelTooOldErrorString)());
    return undefined;
  }
  return (0, _common2.convertSavedPanelsToPanelMap)(appStateInUrl.panels);
}

/**
 * Loads any dashboard state from the URL, and removes the state from the URL.
 */
const loadAndRemoveDashboardState = kbnUrlStateStorage => {
  const rawAppStateInUrl = kbnUrlStateStorage.get(_dashboard_constants.DASHBOARD_STATE_STORAGE_KEY);
  if (!rawAppStateInUrl) return {};
  const panelsMap = getPanelsMap(rawAppStateInUrl);
  const nextUrl = (0, _common.replaceUrlHashQuery)(window.location.href, hashQuery => {
    delete hashQuery[_dashboard_constants.DASHBOARD_STATE_STORAGE_KEY];
    return hashQuery;
  });
  kbnUrlStateStorage.kbnUrlControls.update(nextUrl, true);
  const partialState = {
    ..._lodash.default.omit(rawAppStateInUrl, ['panels', 'query']),
    ...(panelsMap ? {
      panels: panelsMap
    } : {}),
    ...(rawAppStateInUrl.query ? {
      query: (0, _load_dashboard_state.migrateLegacyQuery)(rawAppStateInUrl.query)
    } : {})
  };
  return partialState;
};
exports.loadAndRemoveDashboardState = loadAndRemoveDashboardState;
const startSyncingDashboardUrlState = ({
  kbnUrlStateStorage,
  dashboardAPI
}) => {
  const appStateSubscription = kbnUrlStateStorage.change$(_dashboard_constants.DASHBOARD_STATE_STORAGE_KEY).pipe((0, _operators.debounceTime)(10)) // debounce URL updates so react has time to unsubscribe when changing URLs
  .subscribe(() => {
    const stateFromUrl = loadAndRemoveDashboardState(kbnUrlStateStorage);
    if (Object.keys(stateFromUrl).length === 0) return;
    dashboardAPI.updateInput(stateFromUrl);
  });
  const stopWatchingAppStateInUrl = () => appStateSubscription.unsubscribe();
  return {
    stopWatchingAppStateInUrl
  };
};
exports.startSyncingDashboardUrlState = startSyncingDashboardUrlState;