"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performUpdate = void 0;
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
var _constants = require("../constants");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const performUpdate = async ({
  id,
  type,
  attributes,
  options
}, {
  registry,
  helpers,
  allowedTypes,
  client,
  serializer,
  migrator,
  extensions = {}
}) => {
  var _preflightResult$save, _preflightResult, _preflightResult2, _preflightResult3, _preflightResult4, _body$get$_source, _body$get;
  const {
    common: commonHelper,
    encryption: encryptionHelper,
    preflight: preflightHelper,
    migration: migrationHelper
  } = helpers;
  const {
    securityExtension
  } = extensions;
  const namespace = commonHelper.getCurrentNamespace(options.namespace);
  if (!allowedTypes.includes(type)) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
  }
  if (!id) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('id cannot be empty'); // prevent potentially upserting a saved object with an empty ID
  }

  const {
    version,
    references,
    upsert,
    refresh = _constants.DEFAULT_REFRESH_SETTING,
    retryOnConflict = version ? 0 : _constants.DEFAULT_RETRY_COUNT
  } = options;
  let preflightResult;
  if (registry.isMultiNamespace(type)) {
    preflightResult = await preflightHelper.preflightCheckNamespaces({
      type,
      id,
      namespace
    });
  }
  const existingNamespaces = (_preflightResult$save = (_preflightResult = preflightResult) === null || _preflightResult === void 0 ? void 0 : _preflightResult.savedObjectNamespaces) !== null && _preflightResult$save !== void 0 ? _preflightResult$save : [];
  const authorizationResult = await (securityExtension === null || securityExtension === void 0 ? void 0 : securityExtension.authorizeUpdate({
    namespace,
    object: {
      type,
      id,
      existingNamespaces
    }
  }));
  if (((_preflightResult2 = preflightResult) === null || _preflightResult2 === void 0 ? void 0 : _preflightResult2.checkResult) === 'found_outside_namespace' || !upsert && ((_preflightResult3 = preflightResult) === null || _preflightResult3 === void 0 ? void 0 : _preflightResult3.checkResult) === 'not_found') {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
  }
  if (upsert && ((_preflightResult4 = preflightResult) === null || _preflightResult4 === void 0 ? void 0 : _preflightResult4.checkResult) === 'not_found') {
    // If an upsert would result in the creation of a new object, we need to check for alias conflicts too.
    // This takes an extra round trip to Elasticsearch, but this won't happen often.
    // TODO: improve performance by combining these into a single preflight check
    await preflightHelper.preflightCheckForUpsertAliasConflict(type, id, namespace);
  }
  const time = (0, _utils.getCurrentTime)();
  let rawUpsert;
  // don't include upsert if the object already exists; ES doesn't allow upsert in combination with version properties
  if (upsert && (!preflightResult || preflightResult.checkResult === 'not_found')) {
    let savedObjectNamespace;
    let savedObjectNamespaces;
    if (registry.isSingleNamespace(type) && namespace) {
      savedObjectNamespace = namespace;
    } else if (registry.isMultiNamespace(type)) {
      savedObjectNamespaces = preflightResult.savedObjectNamespaces;
    }
    const migrated = migrationHelper.migrateInputDocument({
      id,
      type,
      ...(savedObjectNamespace && {
        namespace: savedObjectNamespace
      }),
      ...(savedObjectNamespaces && {
        namespaces: savedObjectNamespaces
      }),
      attributes: {
        ...(await encryptionHelper.optionallyEncryptAttributes(type, id, namespace, upsert))
      },
      updated_at: time
    });
    rawUpsert = serializer.savedObjectToRaw(migrated);
  }
  const doc = {
    [type]: await encryptionHelper.optionallyEncryptAttributes(type, id, namespace, attributes),
    updated_at: time,
    ...(Array.isArray(references) && {
      references
    })
  };
  const body = await client.update({
    id: serializer.generateRawId(namespace, type, id),
    index: commonHelper.getIndexForType(type),
    ...(0, _utils.getExpectedVersionProperties)(version),
    refresh,
    retry_on_conflict: retryOnConflict,
    body: {
      doc,
      ...(rawUpsert && {
        upsert: rawUpsert._source
      })
    },
    _source_includes: ['namespace', 'namespaces', 'originId'],
    require_alias: true
  }).catch(err => {
    if (_coreSavedObjectsServer.SavedObjectsErrorHelpers.isEsUnavailableError(err)) {
      throw err;
    }
    if (_coreSavedObjectsServer.SavedObjectsErrorHelpers.isNotFoundError(err)) {
      // see "404s from missing index" above
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
    }
    throw err;
  });
  const {
    originId
  } = (_body$get$_source = (_body$get = body.get) === null || _body$get === void 0 ? void 0 : _body$get._source) !== null && _body$get$_source !== void 0 ? _body$get$_source : {};
  let namespaces = [];
  if (!registry.isNamespaceAgnostic(type)) {
    var _body$get$_source$nam, _body$get2, _body$get3;
    namespaces = (_body$get$_source$nam = (_body$get2 = body.get) === null || _body$get2 === void 0 ? void 0 : _body$get2._source.namespaces) !== null && _body$get$_source$nam !== void 0 ? _body$get$_source$nam : [_coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString((_body$get3 = body.get) === null || _body$get3 === void 0 ? void 0 : _body$get3._source.namespace)];
  }
  const result = {
    id,
    type,
    updated_at: time,
    version: (0, _coreSavedObjectsBaseServerInternal.encodeHitVersion)(body),
    namespaces,
    ...(originId && {
      originId
    }),
    references,
    attributes
  };
  return encryptionHelper.optionallyDecryptAndRedactSingleResult(result, authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.typeMap, attributes);
};
exports.performUpdate = performUpdate;