"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.create = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _server = require("@kbn/core/server");
var _api = require("../../../common/api");
var _constants = require("../../../common/constants");
var _validators = require("../../../common/utils/validators");
var _authorization = require("../../authorization");
var _error = require("../../common/error");
var _utils = require("../../common/utils");
var _constants2 = require("../../common/constants");
var _runtime_types = require("../../../common/api/runtime_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validateCategory(category) {
  if ((0, _validators.isCategoryFieldTooLong)(category)) {
    throw _boom.default.badRequest(`The length of the category is too long. The maximum length is ${_constants.MAX_CATEGORY_LENGTH}`);
  }
  if ((0, _validators.isCategoryFieldInvalidString)(category)) {
    throw _boom.default.badRequest('The category cannot be an empty string.');
  }
}

/**
 * Creates a new case.
 *
 * @ignore
 */
const create = async (data, clientArgs) => {
  const {
    services: {
      caseService,
      userActionService,
      licensingService,
      notificationService
    },
    user,
    logger,
    authorization: auth
  } = clientArgs;
  try {
    var _query$severity, _query$assignees, _query$category;
    const query = (0, _api.decodeWithExcessOrThrow)(_api.CasePostRequestRt)(data);
    if (query.title.length > _constants.MAX_TITLE_LENGTH) {
      throw _boom.default.badRequest(`The length of the title is too long. The maximum length is ${_constants.MAX_TITLE_LENGTH}.`);
    }
    if (query.tags.some(_validators.isInvalidTag)) {
      throw _boom.default.badRequest('A tag must contain at least one non-space character');
    }
    validateCategory(query.category);
    const savedObjectID = _server.SavedObjectsUtils.generateId();
    await auth.ensureAuthorized({
      operation: _authorization.Operations.createCase,
      entities: [{
        owner: query.owner,
        id: savedObjectID
      }]
    });

    /**
     * Assign users to a case is only available to Platinum+
     */

    if (query.assignees && query.assignees.length !== 0) {
      const hasPlatinumLicenseOrGreater = await licensingService.isAtLeastPlatinum();
      if (!hasPlatinumLicenseOrGreater) {
        throw _boom.default.forbidden('In order to assign users to cases, you must be subscribed to an Elastic Platinum license');
      }
      licensingService.notifyUsage(_constants2.LICENSING_CASE_ASSIGNMENT_FEATURE);
    }
    if ((0, _validators.areTotalAssigneesInvalid)(query.assignees)) {
      throw _boom.default.badRequest(`You cannot assign more than ${_constants.MAX_ASSIGNEES_PER_CASE} assignees to a case.`);
    }
    const newCase = await caseService.postNewCase({
      attributes: (0, _utils.transformNewCase)({
        user,
        newCase: query
      }),
      id: savedObjectID,
      refresh: false
    });
    await userActionService.creator.createUserAction({
      type: _api.ActionTypes.create_case,
      caseId: newCase.id,
      user,
      payload: {
        ...query,
        severity: (_query$severity = query.severity) !== null && _query$severity !== void 0 ? _query$severity : _api.CaseSeverity.LOW,
        assignees: (_query$assignees = query.assignees) !== null && _query$assignees !== void 0 ? _query$assignees : [],
        category: (_query$category = query.category) !== null && _query$category !== void 0 ? _query$category : null
      },
      owner: newCase.attributes.owner
    });
    if (query.assignees && query.assignees.length !== 0) {
      const assigneesWithoutCurrentUser = query.assignees.filter(assignee => assignee.uid !== user.profile_uid);
      await notificationService.notifyAssignees({
        assignees: assigneesWithoutCurrentUser,
        theCase: newCase
      });
    }
    const res = (0, _utils.flattenCaseSavedObject)({
      savedObject: newCase
    });
    return (0, _runtime_types.decodeOrThrow)(_api.CaseRt)(res);
  } catch (error) {
    throw (0, _error.createCaseError)({
      message: `Failed to create case: ${error}`,
      error,
      logger
    });
  }
};
exports.create = create;