"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkGet = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _lodash = require("lodash");
var _constants = require("../../../common/constants");
var _api = require("../../../common/api");
var _error = require("../../common/error");
var _utils = require("../../common/utils");
var _authorization = require("../../authorization");
var _runtime_types = require("../../../common/api/runtime_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Retrieves multiple cases by ids.
 */
const bulkGet = async (params, clientArgs) => {
  const {
    services: {
      caseService,
      attachmentService
    },
    logger,
    authorization
  } = clientArgs;
  try {
    const request = (0, _api.decodeWithExcessOrThrow)(_api.CasesBulkGetRequestRt)(params);
    throwErrorIfCaseIdsReachTheLimit(request.ids);
    const cases = await caseService.getCases({
      caseIds: request.ids
    });
    const [validCases, soBulkGetErrors] = (0, _lodash.partition)(cases.saved_objects, caseInfo => caseInfo.error === undefined);
    const {
      authorized: authorizedCases,
      unauthorized: unauthorizedCases
    } = await authorization.getAndEnsureAuthorizedEntities({
      savedObjects: validCases,
      operation: _authorization.Operations.bulkGetCases
    });
    const commentTotals = await attachmentService.getter.getCaseCommentStats({
      caseIds: authorizedCases.map(theCase => theCase.id)
    });
    const flattenedCases = authorizedCases.map(theCase => {
      var _commentTotals$get;
      const {
        userComments,
        alerts
      } = (_commentTotals$get = commentTotals.get(theCase.id)) !== null && _commentTotals$get !== void 0 ? _commentTotals$get : {
        alerts: 0,
        userComments: 0
      };
      return (0, _utils.flattenCaseSavedObject)({
        savedObject: theCase,
        totalComment: userComments,
        totalAlerts: alerts
      });
    });
    const errors = constructErrors(soBulkGetErrors, unauthorizedCases);
    const res = {
      cases: flattenedCases,
      errors
    };
    return (0, _runtime_types.decodeOrThrow)(_api.CasesBulkGetResponseRt)(res);
  } catch (error) {
    var _params$ids;
    const ids = (_params$ids = params.ids) !== null && _params$ids !== void 0 ? _params$ids : [];
    throw (0, _error.createCaseError)({
      message: `Failed to bulk get cases: ${ids.join(', ')}: ${error}`,
      error,
      logger
    });
  }
};
exports.bulkGet = bulkGet;
const throwErrorIfCaseIdsReachTheLimit = ids => {
  if (ids.length > _constants.MAX_BULK_GET_CASES) {
    throw _boom.default.badRequest(`Maximum request limit of ${_constants.MAX_BULK_GET_CASES} cases reached`);
  }
};
const constructErrors = (soBulkGetErrors, unauthorizedCases) => {
  const errors = [];
  for (const soError of soBulkGetErrors) {
    errors.push({
      error: soError.error.error,
      message: soError.error.message,
      status: soError.error.statusCode,
      caseId: soError.id
    });
  }
  for (const theCase of unauthorizedCases) {
    errors.push({
      error: 'Forbidden',
      message: `Unauthorized to access case with owner: "${theCase.attributes.owner}"`,
      status: 403,
      caseId: theCase.id
    });
  }
  return errors;
};