"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.assetsRoutes = assetsRoutes;
var _lodash = require("lodash");
var rt = _interopRequireWildcard(require("io-ts"));
var _ioTsUtils = require("@kbn/io-ts-utils");
var _debug_log = require("../../common/debug_log");
var _types_api = require("../../common/types_api");
var _constants = require("../constants");
var _get_assets = require("../lib/get_assets");
var _get_all_related_assets = require("../lib/get_all_related_assets");
var _utils = require("./utils");
var _errors = require("../lib/errors");
var _utils2 = require("../lib/utils");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const sizeRT = rt.union([(0, _ioTsUtils.inRangeFromStringRt)(1, 100), (0, _ioTsUtils.createLiteralValueFromUndefinedRT)(10)]);
const assetDateRT = rt.union([_ioTsUtils.dateRt, _ioTsUtils.datemathStringRt]);
const getAssetsQueryOptionsRT = rt.exact(rt.partial({
  from: assetDateRT,
  to: assetDateRT,
  type: rt.union([rt.array(_types_api.assetTypeRT), _types_api.assetTypeRT]),
  ean: rt.union([rt.array(rt.string), rt.string]),
  size: sizeRT
}));
const getAssetsDiffQueryOptionsRT = rt.exact(rt.intersection([rt.type({
  aFrom: assetDateRT,
  aTo: assetDateRT,
  bFrom: assetDateRT,
  bTo: assetDateRT
}), rt.partial({
  type: rt.union([rt.array(_types_api.assetTypeRT), _types_api.assetTypeRT])
})]));
const getRelatedAssetsQueryOptionsRT = rt.exact(rt.intersection([rt.type({
  from: assetDateRT,
  ean: rt.string,
  relation: _types_api.relationRT,
  size: sizeRT,
  maxDistance: rt.union([(0, _ioTsUtils.inRangeFromStringRt)(1, 5), (0, _ioTsUtils.createLiteralValueFromUndefinedRT)(1)])
}), rt.partial({
  to: assetDateRT,
  type: rt.union([rt.array(_types_api.assetTypeRT), _types_api.assetTypeRT])
})]));
function assetsRoutes({
  router
}) {
  // GET /assets
  router.get({
    path: `${_constants.ASSET_MANAGER_API_BASE}/assets`,
    validate: {
      query: (0, _ioTsUtils.createRouteValidationFunction)(getAssetsQueryOptionsRT)
    }
  }, async (context, req, res) => {
    const {
      size,
      ...filters
    } = req.query || {};
    if (filters.type && filters.ean) {
      return res.badRequest({
        body: 'Filters "type" and "ean" are mutually exclusive but found both.'
      });
    }
    const esClient = await (0, _utils.getEsClientFromContext)(context);
    try {
      const results = await (0, _get_assets.getAssets)({
        esClient,
        size,
        filters
      });
      return res.ok({
        body: {
          results
        }
      });
    } catch (error) {
      (0, _debug_log.debug)('error looking up asset records', error);
      return res.customError({
        statusCode: 500
      });
    }
  });

  // GET assets/related
  router.get({
    path: `${_constants.ASSET_MANAGER_API_BASE}/assets/related`,
    validate: {
      query: (0, _ioTsUtils.createRouteValidationFunction)(getRelatedAssetsQueryOptionsRT)
    }
  }, async (context, req, res) => {
    // Add references into sample data and write integration tests

    const {
      from,
      to,
      ean,
      relation,
      maxDistance,
      size
    } = req.query || {};
    const esClient = await (0, _utils.getEsClientFromContext)(context);
    const type = (0, _utils2.toArray)(req.query.type);
    if (to && !(0, _utils2.isValidRange)(from, to)) {
      return res.badRequest({
        body: `Time range cannot move backwards in time. "to" (${to}) is before "from" (${from}).`
      });
    }
    try {
      return res.ok({
        body: {
          results: await (0, _get_all_related_assets.getAllRelatedAssets)(esClient, {
            ean,
            from,
            to,
            type,
            maxDistance,
            size,
            relation
          })
        }
      });
    } catch (error) {
      (0, _debug_log.debug)('error looking up asset records', error);
      if (error instanceof _errors.AssetNotFoundError) {
        return res.customError({
          statusCode: 404,
          body: error.message
        });
      }
      return res.customError({
        statusCode: 500,
        body: error.message
      });
    }
  });

  // GET /assets/diff
  router.get({
    path: `${_constants.ASSET_MANAGER_API_BASE}/assets/diff`,
    validate: {
      query: (0, _ioTsUtils.createRouteValidationFunction)(getAssetsDiffQueryOptionsRT)
    }
  }, async (context, req, res) => {
    const {
      aFrom,
      aTo,
      bFrom,
      bTo
    } = req.query;
    const type = (0, _utils2.toArray)(req.query.type);
    if (!(0, _utils2.isValidRange)(aFrom, aTo)) {
      return res.badRequest({
        body: `Time range cannot move backwards in time. "aTo" (${aTo}) is before "aFrom" (${aFrom}).`
      });
    }
    if (!(0, _utils2.isValidRange)(bFrom, bTo)) {
      return res.badRequest({
        body: `Time range cannot move backwards in time. "bTo" (${bTo}) is before "bFrom" (${bFrom}).`
      });
    }
    const esClient = await (0, _utils.getEsClientFromContext)(context);
    try {
      const resultsForA = await (0, _get_assets.getAssets)({
        esClient,
        filters: {
          from: aFrom,
          to: aTo,
          type
        }
      });
      const resultsForB = await (0, _get_assets.getAssets)({
        esClient,
        filters: {
          from: bFrom,
          to: bTo,
          type
        }
      });
      const onlyInA = (0, _lodash.differenceBy)(resultsForA, resultsForB, 'asset.ean');
      const onlyInB = (0, _lodash.differenceBy)(resultsForB, resultsForA, 'asset.ean');
      const inBoth = (0, _lodash.intersectionBy)(resultsForA, resultsForB, 'asset.ean');
      return res.ok({
        body: {
          onlyInA,
          onlyInB,
          inBoth
        }
      });
    } catch (error) {
      (0, _debug_log.debug)('error looking up asset records', error);
      return res.customError({
        statusCode: 500
      });
    }
  });
}