"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.clearCurrentActiveSnooze = clearCurrentActiveSnooze;
exports.clearScheduledSnoozesById = clearScheduledSnoozesById;
exports.clearUnscheduledSnooze = clearUnscheduledSnooze;
exports.getBulkSnoozeAttributes = getBulkSnoozeAttributes;
exports.getBulkUnsnoozeAttributes = getBulkUnsnoozeAttributes;
exports.getSnoozeAttributes = getSnoozeAttributes;
exports.getUnsnoozeAttributes = getUnsnoozeAttributes;
exports.verifySnoozeScheduleLimit = verifySnoozeScheduleLimit;
var _i18n = require("@kbn/i18n");
var _is_rule_snoozed = require("../../lib/is_rule_snoozed");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getSnoozeAttributes(attributes, snoozeSchedule) {
  // If duration is -1, instead mute all
  const {
    id: snoozeId,
    duration
  } = snoozeSchedule;
  if (duration === -1) {
    return {
      muteAll: true,
      snoozeSchedule: clearUnscheduledSnooze(attributes)
    };
  }
  return {
    snoozeSchedule: (snoozeId ? clearScheduledSnoozesById(attributes, [snoozeId]) : clearUnscheduledSnooze(attributes)).concat(snoozeSchedule),
    muteAll: false
  };
}
function getBulkSnoozeAttributes(attributes, snoozeSchedule) {
  // If duration is -1, instead mute all
  const {
    id: snoozeId,
    duration
  } = snoozeSchedule;
  if (duration === -1) {
    return {
      muteAll: true,
      snoozeSchedule: clearUnscheduledSnooze(attributes)
    };
  }

  // Bulk adding snooze schedule, don't touch the existing snooze/indefinite snooze
  if (snoozeId) {
    const existingSnoozeSchedules = attributes.snoozeSchedule || [];
    return {
      muteAll: attributes.muteAll,
      snoozeSchedule: [...existingSnoozeSchedules, snoozeSchedule]
    };
  }

  // Bulk snoozing, don't touch the existing snooze schedules
  return {
    muteAll: false,
    snoozeSchedule: [...clearUnscheduledSnooze(attributes), snoozeSchedule]
  };
}
function getUnsnoozeAttributes(attributes, scheduleIds) {
  const snoozeSchedule = scheduleIds ? clearScheduledSnoozesById(attributes, scheduleIds) : clearCurrentActiveSnooze(attributes);
  return {
    snoozeSchedule,
    ...(!scheduleIds ? {
      muteAll: false
    } : {})
  };
}
function getBulkUnsnoozeAttributes(attributes, scheduleIds) {
  // Bulk removing snooze schedules, don't touch the current snooze/indefinite snooze
  if (scheduleIds) {
    var _attributes$snoozeSch;
    const newSchedules = clearScheduledSnoozesById(attributes, scheduleIds);
    // Unscheduled snooze is also known as snooze now
    const unscheduledSnooze = ((_attributes$snoozeSch = attributes.snoozeSchedule) === null || _attributes$snoozeSch === void 0 ? void 0 : _attributes$snoozeSch.filter(s => typeof s.id === 'undefined')) || [];
    return {
      snoozeSchedule: [...unscheduledSnooze, ...newSchedules],
      muteAll: attributes.muteAll
    };
  }

  // Bulk unsnoozing, don't touch current snooze schedules that are NOT active
  return {
    snoozeSchedule: clearCurrentActiveSnooze(attributes),
    muteAll: false
  };
}
function clearUnscheduledSnooze(attributes) {
  // Clear any snoozes that have no ID property. These are "simple" snoozes created with the quick UI, e.g. snooze for 3 days starting now
  return attributes.snoozeSchedule ? attributes.snoozeSchedule.filter(s => typeof s.id !== 'undefined') : [];
}
function clearScheduledSnoozesById(attributes, ids) {
  return attributes.snoozeSchedule ? attributes.snoozeSchedule.filter(s => s.id && !ids.includes(s.id)) : [];
}
function clearCurrentActiveSnooze(attributes) {
  var _activeSnoozes$map;
  // First attempt to cancel a simple (unscheduled) snooze
  const clearedUnscheduledSnoozes = clearUnscheduledSnooze(attributes);
  // Now clear any scheduled snoozes that are currently active and never recur
  const activeSnoozes = (0, _is_rule_snoozed.getActiveScheduledSnoozes)(attributes);
  const activeSnoozeIds = (_activeSnoozes$map = activeSnoozes === null || activeSnoozes === void 0 ? void 0 : activeSnoozes.map(s => s.id)) !== null && _activeSnoozes$map !== void 0 ? _activeSnoozes$map : [];
  const recurringSnoozesToSkip = [];
  const clearedNonRecurringActiveSnoozes = clearedUnscheduledSnoozes.filter(s => {
    if (!activeSnoozeIds.includes(s.id)) return true;
    // Check if this is a recurring snooze, and return true if so
    if (s.rRule.freq && s.rRule.count !== 1) {
      recurringSnoozesToSkip.push(s.id);
      return true;
    }
  });
  const clearedSnoozesAndSkippedRecurringSnoozes = clearedNonRecurringActiveSnoozes.map(s => {
    var _activeSnoozes$find, _s$skipRecurrences;
    if (s.id && !recurringSnoozesToSkip.includes(s.id)) return s;
    const currentRecurrence = activeSnoozes === null || activeSnoozes === void 0 ? void 0 : (_activeSnoozes$find = activeSnoozes.find(a => a.id === s.id)) === null || _activeSnoozes$find === void 0 ? void 0 : _activeSnoozes$find.lastOccurrence;
    if (!currentRecurrence) return s;
    return {
      ...s,
      skipRecurrences: ((_s$skipRecurrences = s.skipRecurrences) !== null && _s$skipRecurrences !== void 0 ? _s$skipRecurrences : []).concat(currentRecurrence.toISOString())
    };
  });
  return clearedSnoozesAndSkippedRecurringSnoozes;
}
function verifySnoozeScheduleLimit(attributes) {
  var _attributes$snoozeSch2;
  const schedules = (_attributes$snoozeSch2 = attributes.snoozeSchedule) === null || _attributes$snoozeSch2 === void 0 ? void 0 : _attributes$snoozeSch2.filter(snooze => snooze.id);
  if (schedules && schedules.length > 5) {
    throw Error(_i18n.i18n.translate('xpack.alerting.rulesClient.snoozeSchedule.limitReached', {
      defaultMessage: 'Rule cannot have more than 5 snooze schedules'
    }));
  }
}