"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useCategorizeRequest = useCategorizeRequest;
var _lodash = require("lodash");
var _react = require("react");
var _public = require("@kbn/data-plugin/public");
var _mlLocalStorage = require("@kbn/ml-local-storage");
var _use_aiops_app_context = require("../../hooks/use_aiops_app_context");
var _sampling_menu = require("./sampling_menu");
var _storage = require("../../types/storage");
var _random_sampler = require("./sampling_menu/random_sampler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CATEGORY_LIMIT = 1000;
const EXAMPLE_LIMIT = 1;
function useCategorizeRequest() {
  const [randomSamplerMode, setRandomSamplerMode] = (0, _mlLocalStorage.useStorage)(_storage.AIOPS_RANDOM_SAMPLING_MODE_PREFERENCE, _random_sampler.RANDOM_SAMPLER_OPTION.ON_AUTOMATIC);
  const [randomSamplerProbability, setRandomSamplerProbability] = (0, _mlLocalStorage.useStorage)(_storage.AIOPS_RANDOM_SAMPLING_PROBABILITY_PREFERENCE, _random_sampler.DEFAULT_PROBABILITY);
  const {
    data
  } = (0, _use_aiops_app_context.useAiopsAppContext)();
  const abortController = (0, _react.useRef)(new AbortController());
  const randomSampler = (0, _react.useMemo)(() => new _sampling_menu.RandomSampler(randomSamplerMode, setRandomSamplerMode, randomSamplerProbability, setRandomSamplerProbability),
  // eslint-disable-next-line react-hooks/exhaustive-deps
  []);
  const runCategorizeRequest = (0, _react.useCallback)((index, field, timeField, from, to, query, intervalMs) => {
    const {
      wrap,
      unwrap
    } = randomSampler.createRandomSamplerWrapper();
    return new Promise((resolve, reject) => {
      data.search.search(createCategoryRequest(index, field, timeField, from, to, query, wrap, intervalMs), {
        abortSignal: abortController.current.signal
      }).subscribe({
        next: result => {
          if ((0, _public.isCompleteResponse)(result)) {
            resolve(processCategoryResults(result, field, unwrap));
          } else if ((0, _public.isErrorResponse)(result)) {
            reject(result);
          } else {
            // partial results
            // Ignore partial results for now.
            // An issue with the search function means partial results are not being returned correctly.
          }
        },
        error: error => {
          if (error.name === 'AbortError') {
            return resolve({
              categories: [],
              sparkLinesPerCategory: {}
            });
          }
          reject(error);
        }
      });
    });
  }, [data.search, randomSampler]);
  const cancelRequest = (0, _react.useCallback)(() => {
    abortController.current.abort();
    abortController.current = new AbortController();
  }, []);
  return {
    runCategorizeRequest,
    cancelRequest,
    randomSampler
  };
}
function createCategoryRequest(index, field, timeField, from, to, queryIn, wrap, intervalMs) {
  const query = (0, _lodash.cloneDeep)(queryIn);
  if (query.bool === undefined) {
    query.bool = {};
  }
  if (query.bool.must === undefined) {
    query.bool.must = [];
    if (query.match_all !== undefined) {
      query.bool.must.push({
        match_all: query.match_all
      });
      delete query.match_all;
    }
  }
  if (query.multi_match !== undefined) {
    query.bool.should = {
      multi_match: query.multi_match
    };
    delete query.multi_match;
  }
  query.bool.must.push({
    range: {
      [timeField]: {
        gte: from,
        lte: to,
        format: 'epoch_millis'
      }
    }
  });
  const aggs = {
    categories: {
      categorize_text: {
        field,
        size: CATEGORY_LIMIT
      },
      aggs: {
        hit: {
          top_hits: {
            size: EXAMPLE_LIMIT,
            sort: [timeField],
            _source: field
          }
        },
        ...(intervalMs ? {
          sparkline: {
            date_histogram: {
              field: timeField,
              fixed_interval: `${intervalMs}ms`
            }
          }
        } : {})
      }
    }
  };
  return {
    params: {
      index,
      size: 0,
      body: {
        query,
        aggs: wrap(aggs)
      }
    }
  };
}
function processCategoryResults(result, field, unwrap) {
  const sparkLinesPerCategory = {};
  const {
    aggregations
  } = result.rawResponse;
  if (aggregations === undefined) {
    throw new Error('processCategoryResults failed, did not return aggregations.');
  }
  const {
    categories: {
      buckets
    }
  } = unwrap(aggregations);
  const categories = buckets.map(b => {
    sparkLinesPerCategory[b.key] = b.sparkline === undefined ? {} : b.sparkline.buckets.reduce((acc2, cur2) => {
      acc2[cur2.key] = cur2.doc_count;
      return acc2;
    }, {});
    return {
      key: b.key,
      count: b.doc_count,
      examples: b.hit.hits.hits.map(h => (0, _lodash.get)(h._source, field))
    };
  });
  return {
    categories,
    sparkLinesPerCategory
  };
}