"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useEuiButtonRadiusCSS = exports.useEuiButtonFocusCSS = exports.useEuiButtonColorCSS = exports.euiButtonFillColor = exports.euiButtonEmptyColor = exports.euiButtonColor = exports.BUTTON_COLORS = void 0;
var _react = require("@emotion/react");
var _global_styling = require("../../../../global_styling");
var _services = require("../../../../services");
function _toConsumableArray(arr) { return _arrayWithoutHoles(arr) || _iterableToArray(arr) || _unsupportedIterableToArray(arr) || _nonIterableSpread(); }
function _nonIterableSpread() { throw new TypeError("Invalid attempt to spread non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(o, minLen) { if (!o) return; if (typeof o === "string") return _arrayLikeToArray(o, minLen); var n = Object.prototype.toString.call(o).slice(8, -1); if (n === "Object" && o.constructor) n = o.constructor.name; if (n === "Map" || n === "Set") return Array.from(o); if (n === "Arguments" || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(n)) return _arrayLikeToArray(o, minLen); }
function _iterableToArray(iter) { if (typeof Symbol !== "undefined" && iter[Symbol.iterator] != null || iter["@@iterator"] != null) return Array.from(iter); }
function _arrayWithoutHoles(arr) { if (Array.isArray(arr)) return _arrayLikeToArray(arr); }
function _arrayLikeToArray(arr, len) { if (len == null || len > arr.length) len = arr.length; for (var i = 0, arr2 = new Array(len); i < len; i++) arr2[i] = arr[i]; return arr2; } /*
                                                                                                                                                                                     * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
                                                                                                                                                                                     * or more contributor license agreements. Licensed under the Elastic License
                                                                                                                                                                                     * 2.0 and the Server Side Public License, v 1; you may not use this file except
                                                                                                                                                                                     * in compliance with, at your election, the Elastic License 2.0 or the Server
                                                                                                                                                                                     * Side Public License, v 1.
                                                                                                                                                                                     */
var BUTTON_COLORS = ['text', 'accent', 'primary', 'success', 'warning', 'danger'];
exports.BUTTON_COLORS = BUTTON_COLORS;
/**
 * Creates the `base` version of button styles with proper text contrast.
 * @param euiThemeContext
 * @param color One of the named button colors or 'disabled'
 * @returns Style object `{ backgroundColor, color }`
 */
var euiButtonColor = function euiButtonColor(euiThemeContext, color) {
  var euiTheme = euiThemeContext.euiTheme,
    colorMode = euiThemeContext.colorMode;
  function tintOrShade(color) {
    return colorMode === 'DARK' ? (0, _services.shade)(color, 0.7) : (0, _services.tint)(color, 0.8);
  }
  var foreground;
  var background;
  switch (color) {
    case 'disabled':
      return {
        color: euiTheme.colors.disabledText,
        backgroundColor: (0, _services.transparentize)(euiTheme.colors.lightShade, 0.15)
      };
    case 'text':
      foreground = euiTheme.colors[color];
      background = colorMode === 'DARK' ? (0, _services.shade)(euiTheme.colors.lightShade, 0.2) : (0, _services.tint)(euiTheme.colors.lightShade, 0.5);
      break;
    default:
      foreground = euiTheme.colors["".concat(color, "Text")];
      background = tintOrShade(euiTheme.colors[color]);
      break;
  }
  return {
    color: (0, _services.makeHighContrastColor)(foreground)(background),
    backgroundColor: background
  };
};

/**
 * Creates the `fill` version of buttons styles with proper text contrast.
 * @param euiThemeContext
 * @param color One of the named button colors or 'disabled'
 * @returns Style object `{ backgroundColor, color }`
 */
exports.euiButtonColor = euiButtonColor;
var euiButtonFillColor = function euiButtonFillColor(euiThemeContext, color) {
  var euiTheme = euiThemeContext.euiTheme,
    colorMode = euiThemeContext.colorMode;
  var getForegroundColor = function getForegroundColor(background) {
    return _services.isColorDark.apply(void 0, _toConsumableArray((0, _services.hexToRgb)(background))) ? euiTheme.colors.ghost : euiTheme.colors.ink;
  };
  var background;
  var foreground;
  switch (color) {
    case 'disabled':
      background = euiButtonColor(euiThemeContext, color).backgroundColor;
      foreground = euiButtonColor(euiThemeContext, color).color;
      break;
    case 'text':
      background = colorMode === 'DARK' ? euiTheme.colors.text : euiTheme.colors.darkShade;
      foreground = getForegroundColor(background);
      break;
    case 'success':
    case 'accent':
      // Success / accent fills are hard to read on light mode even though they pass color contrast ratios
      // TODO: If WCAG 3 gets adopted (which would calculates luminosity & would allow us to use white text instead),
      // we can get rid of this case (https://blog.datawrapper.de/color-contrast-check-data-vis-wcag-apca/)
      background = colorMode === 'LIGHT' ? (0, _services.tint)(euiTheme.colors[color], 0.3) : euiTheme.colors[color];
      foreground = getForegroundColor(background);
      break;
    default:
      background = euiTheme.colors[color];
      foreground = getForegroundColor(background);
      break;
  }
  return {
    color: foreground,
    backgroundColor: background
  };
};

/**
 * Creates the `empty` version of button styles using the text-variant and adding interactive styles.
 * @param euiThemeContext
 * @param color One of the named button colors or 'disabled'
 * @returns Style object `{ backgroundColor, color }` where `background` is typically used for interactive states
 */
exports.euiButtonFillColor = euiButtonFillColor;
var euiButtonEmptyColor = function euiButtonEmptyColor(euiThemeContext, color) {
  var foreground;
  var background;
  switch (color) {
    case 'disabled':
      foreground = euiButtonColor(euiThemeContext, color).color;
      background = 'transparent';
      break;
    case 'text':
      foreground = euiButtonColor(euiThemeContext, color).color;
      background = (0, _global_styling.euiBackgroundColor)(euiThemeContext, 'subdued', {
        method: 'transparent'
      });
      break;
    default:
      foreground = euiButtonColor(euiThemeContext, color).color;
      background = (0, _global_styling.euiBackgroundColor)(euiThemeContext, color, {
        method: 'transparent'
      });
      break;
  }
  return {
    color: foreground,
    backgroundColor: background
  };
};

/**
 * Given the button display type, returns the Emotion based color keys.
 * @param options Button display type
 * @returns An object of `_EuiButtonColor` keys including `disabled`
 */
exports.euiButtonEmptyColor = euiButtonEmptyColor;
var useEuiButtonColorCSS = function useEuiButtonColorCSS() {
  var options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
  var euiThemeContext = (0, _services.useEuiTheme)();
  function stylesByDisplay(color) {
    return {
      base: /*#__PURE__*/(0, _react.css)(euiButtonColor(euiThemeContext, color), ";;label:base;"),
      fill: /*#__PURE__*/(0, _react.css)(euiButtonFillColor(euiThemeContext, color), "outline-color:", euiThemeContext.colorMode === 'DARK' && color === 'text' ? 'currentColor' : euiThemeContext.euiTheme.colors.fullShade, ";;label:fill;"),
      empty: /*#__PURE__*/(0, _react.css)("color:", euiButtonEmptyColor(euiThemeContext, color).color, ";&:focus,&:active{background-color:", euiButtonEmptyColor(euiThemeContext, color).backgroundColor, ";};label:empty;")
    };
  }
  return {
    text: /*#__PURE__*/(0, _react.css)(stylesByDisplay('text')[options.display || 'base'], ";label:text;"),
    accent: /*#__PURE__*/(0, _react.css)(stylesByDisplay('accent')[options.display || 'base'], ";label:accent;"),
    primary: /*#__PURE__*/(0, _react.css)(stylesByDisplay('primary')[options.display || 'base'], ";label:primary;"),
    success: /*#__PURE__*/(0, _react.css)(stylesByDisplay('success')[options.display || 'base'], ";label:success;"),
    warning: /*#__PURE__*/(0, _react.css)(stylesByDisplay('warning')[options.display || 'base'], ";label:warning;"),
    danger: /*#__PURE__*/(0, _react.css)(stylesByDisplay('danger')[options.display || 'base'], ";label:danger;"),
    disabled: /*#__PURE__*/(0, _react.css)(stylesByDisplay('disabled')[options.display || 'base'], ";label:disabled;")
  };
};

/**
 * Based on the button size, creates the style properties.
 * @returns An object of button size keys with Emption styles for `border-radius`
 */
exports.useEuiButtonColorCSS = useEuiButtonColorCSS;
var useEuiButtonRadiusCSS = function useEuiButtonRadiusCSS() {
  var _useEuiTheme = (0, _services.useEuiTheme)(),
    euiTheme = _useEuiTheme.euiTheme;
  return {
    xs: /*#__PURE__*/(0, _react.css)("border-radius:", euiTheme.border.radius.small, ";;label:xs;"),
    s: /*#__PURE__*/(0, _react.css)("border-radius:", euiTheme.border.radius.small, ";;label:s;"),
    m: /*#__PURE__*/(0, _react.css)("border-radius:", euiTheme.border.radius.medium, ";;label:m;")
  };
};

/**
 * Creates the translate animation when button is in focus.
 * @returns string
 */
exports.useEuiButtonRadiusCSS = useEuiButtonRadiusCSS;
var useEuiButtonFocusCSS = function useEuiButtonFocusCSS() {
  var _useEuiTheme2 = (0, _services.useEuiTheme)(),
    euiTheme = _useEuiTheme2.euiTheme;
  return "\n    ".concat(_global_styling.euiCanAnimate, " {\n      transition: transform ").concat(euiTheme.animation.normal, " ease-in-out,\n        background-color ").concat(euiTheme.animation.normal, " ease-in-out;\n\n      &:hover:not(:disabled) {\n        transform: translateY(-1px);\n      }\n\n      &:focus {\n        animation: euiButtonActive ").concat(euiTheme.animation.normal, "\n          ").concat(euiTheme.animation.bounce, ";\n      }\n\n      &:active:not(:disabled) {\n        transform: translateY(1px);\n      }\n    }\n  ");
};
exports.useEuiButtonFocusCSS = useEuiButtonFocusCSS;