"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PointSeries = void 0;
var _d = _interopRequireDefault(require("d3"));
var _lodash = _interopRequireDefault(require("lodash"));
var _jquery = _interopRequireDefault(require("jquery"));
var _tooltip = require("../components/tooltip");
var _chart = require("./_chart");
var _time_marker = require("./time_marker");
var _touchdown_template = require("../partials/touchdown_template");
var _heatmap_chart = require("./point_series/heatmap_chart");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Line Chart Visualization
 *
 * @class PointSeries
 * @constructor
 * @extends Chart
 * @param handler {Object} Reference to the Handler Class Constructor
 * @param el {HTMLElement} HTML element to which the chart will be appended
 * @param chartData {Object} Elasticsearch query results for this specific chart
 */
class PointSeries extends _chart.Chart {
  constructor(handler, chartEl, chartData, uiSettings) {
    super(handler, chartEl, chartData, uiSettings);
    this.uiSettings = uiSettings;
    this.handler = handler;
    this.chartData = chartData;
    this.chartEl = chartEl;
    this.chartConfig = this.findChartConfig();
    this.handler.pointSeries = this;
  }
  findChartConfig() {
    const charts = this.handler.visConfig.get('charts');
    const chartIndex = this.handler.data.chartData().indexOf(this.chartData);
    return charts[chartIndex];
  }
  getSeries(seriesId) {
    return this.series.find(series => series.chartData.aggId === seriesId);
  }
  addBackground(svg, width, height) {
    const startX = 0;
    const startY = 0;
    return svg.append('rect').attr('x', startX).attr('y', startY).attr('width', width).attr('height', height).attr('fill', 'transparent').attr('class', 'background');
  }
  addGrid(svg) {
    const {
      width,
      height
    } = svg.node().getBBox();
    return svg.append('g').attr('class', 'grid').call(this.handler.grid.draw(width, height));
  }
  addClipPath(svg) {
    const {
      width,
      height
    } = svg.node().getBBox();
    const startX = 0;
    const startY = 0;
    this.clipPathId = 'chart-area' + _lodash.default.uniqueId();

    // Creating clipPath
    return svg.append('clipPath').attr('id', this.clipPathId).append('rect').attr('x', startX).attr('y', startY).attr('width', width).attr('height', height);
  }
  addEvents(svg) {
    const isBrushable = this.events.isBrushable();
    if (isBrushable) {
      this.events.addBrushEvent(svg);
    }
  }
  createEndZones(svg) {
    const self = this;
    const xAxis = this.handler.categoryAxes[0];
    const xScale = xAxis.getScale();
    const ordered = xAxis.ordered;
    const isHorizontal = xAxis.axisConfig.isHorizontal();
    const missingMinMax = !ordered || _lodash.default.isUndefined(ordered.min) || _lodash.default.isUndefined(ordered.max);
    if (missingMinMax || ordered.endzones === false) return;
    const {
      width,
      height
    } = svg.node().getBBox();

    // we don't want to draw endzones over our min and max values, they
    // are still a part of the dataset. We want to start the endzones just
    // outside of them so we will use these values rather than ordered.min/max
    const oneUnit = (ordered.units || _lodash.default.identity)(1);
    const drawInverted = isHorizontal || xAxis.axisConfig.get('scale.inverted', false);
    const size = isHorizontal ? width : height;
    // points on this axis represent the amount of time they cover,
    // so draw the endzones at the actual time bounds
    const leftEndzone = {
      x: drawInverted ? 0 : Math.max(xScale(ordered.min), 0),
      w: drawInverted ? Math.max(xScale(ordered.min), 0) : height - Math.max(xScale(ordered.min), 0)
    };
    const expandLastBucket = xAxis.axisConfig.get('scale.expandLastBucket');
    const rightLastVal = expandLastBucket ? ordered.max : Math.min(ordered.max, _lodash.default.last(xAxis.values));
    const rightStart = rightLastVal + oneUnit;
    const rightEndzone = {
      x: drawInverted ? xScale(rightStart) : 0,
      w: drawInverted ? Math.max(size - xScale(rightStart), 0) : xScale(rightStart)
    };
    this.endzones = svg.selectAll('.layer').data([leftEndzone, rightEndzone]).enter().insert('g', '.brush').attr('class', 'endzone').append('rect').attr('class', 'zone').attr('x', d => isHorizontal ? d.x : 0).attr('y', d => isHorizontal ? 0 : d.x).attr('height', d => isHorizontal ? height : d.w).attr('width', d => isHorizontal ? d.w : width);
    function callPlay(event) {
      const boundData = event.target.__data__;
      const mouseChartXCoord = event.clientX - self.chartEl.getBoundingClientRect().left;
      const mouseChartYCoord = event.clientY - self.chartEl.getBoundingClientRect().top;
      const wholeBucket = boundData && boundData.x != null;

      // the min and max that the endzones start in
      const min = drawInverted ? leftEndzone.w : rightEndzone.w;
      const max = drawInverted ? rightEndzone.x : leftEndzone.x;

      // bounds of the cursor to consider
      let xLeft = isHorizontal ? mouseChartXCoord : mouseChartYCoord;
      let xRight = isHorizontal ? mouseChartXCoord : mouseChartYCoord;
      if (wholeBucket) {
        xLeft = xScale(boundData.x);
        xRight = xScale(xAxis.addInterval(boundData.x));
      }
      return {
        wholeBucket: wholeBucket,
        touchdown: min > xLeft || max < xRight
      };
    }
    function textFormatter() {
      return (0, _touchdown_template.touchdownTemplate)(callPlay(_d.default.event));
    }
    const endzoneTT = new _tooltip.Tooltip('endzones', this.handler.el, textFormatter, null);
    this.tooltips.push(endzoneTT);
    endzoneTT.order = 0;
    endzoneTT.showCondition = function inEndzone() {
      return callPlay(_d.default.event).touchdown;
    };
    endzoneTT.render()(svg);
  }
  calculateRadiusLimits(data) {
    this.radii = (0, _lodash.default)(data.series).map(function (series) {
      return _lodash.default.map(series.values, 'z');
    }).flattenDeep().reduce(function (result, val) {
      if (result.min > val) result.min = val;
      if (result.max < val) result.max = val;
      return result;
    }, {
      min: Infinity,
      max: -Infinity
    });
  }
  draw() {
    const self = this;
    const $elem = (0, _jquery.default)(this.chartEl);
    const width = this.chartConfig.width = $elem.width();
    const height = this.chartConfig.height = $elem.height();
    const xScale = this.handler.categoryAxes[0].getScale();
    const addTimeMarker = this.chartConfig.addTimeMarker;
    const times = this.chartConfig.times || [];
    let div;
    let svg;
    return function (selection) {
      selection.each(function (data) {
        const el = this;
        div = _d.default.select(el);
        svg = div.append('svg').attr('focusable', 'false').attr('width', width).attr('height', height);
        self.addBackground(svg, width, height);
        self.addGrid(svg);
        self.addClipPath(svg);
        self.addEvents(svg);
        self.createEndZones(svg);
        self.calculateRadiusLimits(data);
        self.series = [];
        _lodash.default.each(self.chartConfig.series, (seriArgs, i) => {
          if (!seriArgs.show) return;
          const series = new _heatmap_chart.HeatmapChart(self.handler, svg, data.series[i], seriArgs, self.uiSettings);
          series.events = self.events;
          svg.call(series.draw());
          self.series.push(series);
        });
        if (addTimeMarker) {
          //Domain end of 'now' will be milliseconds behind current time
          //Extend toTime by 1 minute to ensure those cases have a TimeMarker
          const toTime = new Date(xScale.domain()[1].getTime() + 60000);
          const currentTime = new Date();
          if (toTime > currentTime) {
            new _time_marker.TimeMarker(times, xScale, height).render(svg);
          }
        }
        return svg;
      });
    };
  }
}
exports.PointSeries = PointSeries;