"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Chart = void 0;
var _d = _interopRequireDefault(require("d3"));
var _lodash = _interopRequireDefault(require("lodash"));
var _data_label = require("../lib/_data_label");
var _dispatch = require("../lib/dispatch");
var _services = require("../../services");
var _tooltip = require("../components/tooltip");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * The Base Class for all visualizations.
 *
 * @class Chart
 * @constructor
 * @param handler {Object} Reference to the Handler Class Constructor
 * @param el {HTMLElement} HTML element to which the chart will be appended
 * @param chartData {Object} Elasticsearch query results for this specific chart
 */
class Chart {
  constructor(handler, element, chartData, uiSettings) {
    this.handler = handler;
    this.chartEl = element;
    this.chartData = chartData;
    this.tooltips = [];
    const events = this.events = new _dispatch.Dispatch(handler, uiSettings);
    const fieldFormatter = (0, _services.getFormatService)().deserialize(this.handler.data.get('tooltipFormatter'));
    const tooltipFormatterProvider = _tooltip.pointSeriesTooltipFormatter;
    const tooltipFormatter = tooltipFormatterProvider(fieldFormatter);
    if (this.handler.visConfig && this.handler.visConfig.get('addTooltip', false)) {
      const element = this.handler.el;

      // Add tooltip
      this.tooltip = new _tooltip.Tooltip('chart', element, tooltipFormatter, events, uiSettings);
      this.tooltips.push(this.tooltip);
    }
    this._addIdentifier = _lodash.default.bind(this._addIdentifier, this);
  }

  /**
   * Renders the chart(s)
   *
   * @method render
   * @returns {HTMLElement} Contains the D3 chart
   */
  render() {
    const selection = _d.default.select(this.chartEl);
    selection.selectAll('*').remove();
    selection.call(this.draw());
  }

  /**
   * Append the data label to the element
   *
   * @method _addIdentifier
   * @param selection {Object} d3 select object
   */
  _addIdentifier(selection, labelProp) {
    labelProp = labelProp || 'label';
    const labels = this.handler.data.labels;
    function resolveLabel(datum) {
      if (labels.length === 1) return labels[0];
      if (datum[0]) return datum[0][labelProp];
      return datum[labelProp];
    }
    selection.each(function (datum) {
      const label = resolveLabel(datum);
      if (label != null) (0, _data_label.dataLabel)(this, label);
    });
  }

  /**
   * Removes all DOM elements from the root element
   *
   * @method destroy
   */
  destroy() {
    const selection = _d.default.select(this.chartEl);
    this.events.removeAllListeners();
    this.tooltips.forEach(function (tooltip) {
      tooltip.destroy();
    });
    selection.remove();
  }
}
exports.Chart = Chart;