"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReindexProgress = void 0;
var _react = _interopRequireDefault(require("react"));
var _eui = require("@elastic/eui");
var _i18nReact = require("@kbn/i18n-react");
var _types = require("../../../../../../../common/types");
var _types2 = require("../../../../types");
var _step_progress = require("./step_progress");
var _utils = require("../../../../../lib/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ErrorCallout = ({
  errorMessage
}) => /*#__PURE__*/_react.default.createElement(_eui.EuiCallOut, {
  color: "danger",
  title: "There was an error"
}, /*#__PURE__*/_react.default.createElement(_eui.EuiText, null, /*#__PURE__*/_react.default.createElement("p", null, errorMessage)));
const PausedCallout = () => /*#__PURE__*/_react.default.createElement(_eui.EuiCallOut, {
  color: "warning",
  title: "This step was paused due to a Kibana restart. Click 'Resume' below to continue."
});
const ReindexingDocumentsStepTitle = ({
  reindexState: {
    lastCompletedStep,
    status,
    cancelLoadingState
  },
  cancelReindex
}) => {
  if (status === _types.ReindexStatus.cancelled) {
    return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.cancelledTitle",
      defaultMessage: "Reindexing cancelled."
    }));
  }

  // step is in progress after the new index is created and while it's not completed yet
  const stepInProgress = status === _types.ReindexStatus.inProgress && (lastCompletedStep === _types.ReindexStep.newIndexCreated || lastCompletedStep === _types.ReindexStep.reindexStarted);
  // but the reindex can only be cancelled after it has started
  const showCancelLink = status === _types.ReindexStatus.inProgress && lastCompletedStep === _types.ReindexStep.reindexStarted;
  let cancelText;
  switch (cancelLoadingState) {
    case _types2.CancelLoadingState.Requested:
    case _types2.CancelLoadingState.Loading:
      cancelText = /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.cancelButton.cancellingLabel",
        defaultMessage: "Cancelling\u2026"
      });
      break;
    case _types2.CancelLoadingState.Success:
      cancelText = /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.cancelButton.cancelledLabel",
        defaultMessage: "Cancelled"
      });
      break;
    case _types2.CancelLoadingState.Error:
      cancelText = /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.cancelButton.errorLabel",
        defaultMessage: "Could not cancel"
      });
      break;
    default:
      cancelText = /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.cancelButton.cancelLabel",
        defaultMessage: "Cancel"
      });
  }
  return /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    component: "span"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, stepInProgress ? /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.inProgress.reindexingDocumentsStepTitle",
    defaultMessage: "Reindexing documents."
  }) : /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.reindexingDocumentsStepTitle",
    defaultMessage: "Reindex documents."
  })), showCancelLink && /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, /*#__PURE__*/_react.default.createElement(_eui.EuiLink, {
    "data-test-subj": "cancelReindexingDocumentsButton",
    onClick: cancelReindex,
    disabled: cancelLoadingState !== undefined
  }, cancelText)));
};
const orderedSteps = Object.values(_types.ReindexStep).sort();
const getStepTitle = (step, meta, inProgress) => {
  if (step === _types.ReindexStep.readonly) {
    return inProgress ? /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.inProgress.readonlyStepTitle",
      defaultMessage: "Setting {indexName} index to read-only.",
      values: {
        indexName: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, meta.indexName)
      }
    }) : /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.readonlyStepTitle",
      defaultMessage: "Set {indexName} index to read-only.",
      values: {
        indexName: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, meta.indexName)
      }
    });
  }
  if (step === _types.ReindexStep.newIndexCreated) {
    return inProgress ? /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.inProgress.createIndexStepTitle",
      defaultMessage: "Creating {reindexName} index.",
      values: {
        reindexName: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, meta.reindexName)
      }
    }) : /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.createIndexStepTitle",
      defaultMessage: "Create {reindexName} index.",
      values: {
        reindexName: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, meta.reindexName)
      }
    });
  }
  if (step === _types.ReindexStep.aliasCreated) {
    return inProgress ? /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.inProgress.aliasCreatedStepTitle",
      defaultMessage: "Creating {indexName} alias for {reindexName} index.",
      values: {
        indexName: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, meta.indexName),
        reindexName: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, meta.reindexName)
      }
    }) : /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.aliasCreatedStepTitle",
      defaultMessage: "Create {indexName} alias for {reindexName} index.",
      values: {
        indexName: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, meta.indexName),
        reindexName: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, meta.reindexName)
      }
    });
  }
  if (step === _types.ReindexStep.originalIndexDeleted) {
    return inProgress ? /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.inProgress.originalIndexDeletedStepTitle",
      defaultMessage: "Deleting original {indexName} index.",
      values: {
        indexName: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, meta.indexName)
      }
    }) : /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.originalIndexDeletedStepTitle",
      defaultMessage: "Delete original {indexName} index.",
      values: {
        indexName: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, meta.indexName)
      }
    });
  }
  if (step === _types.ReindexStep.existingAliasesUpdated) {
    return inProgress ? /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.inProgress.aliasesUpdatedStepTitle",
      defaultMessage: "Updating {existingAliases} aliases to point to {reindexName} index.",
      values: {
        existingAliases: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, `[${meta.aliases.join(',')}]`),
        reindexName: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, meta.reindexName)
      }
    }) : /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklist.aliasesUpdatedStepTitle",
      defaultMessage: "Update {existingAliases} aliases to point to {reindexName} index.",
      values: {
        existingAliases: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, `[${meta.aliases.join(',')}]`),
        reindexName: /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, meta.reindexName)
      }
    });
  }
};
/**
 * Displays a list of steps in the reindex operation, the current status, a progress bar,
 * and any error messages that are encountered.
 */
const ReindexProgress = props => {
  const {
    errorMessage,
    lastCompletedStep = -1,
    status,
    reindexTaskPercComplete,
    meta
  } = props.reindexState;
  const getProgressStep = thisStep => {
    const previousStep = orderedSteps[orderedSteps.indexOf(thisStep) - 1];
    if (status === _types.ReindexStatus.failed && lastCompletedStep === previousStep) {
      return {
        title: getStepTitle(thisStep, meta),
        status: 'failed',
        children: /*#__PURE__*/_react.default.createElement(ErrorCallout, {
          errorMessage
        })
      };
    } else if (status === _types.ReindexStatus.paused && lastCompletedStep === previousStep) {
      return {
        title: getStepTitle(thisStep, meta),
        status: 'paused',
        children: /*#__PURE__*/_react.default.createElement(PausedCallout, null)
      };
    } else if (status === _types.ReindexStatus.cancelled && lastCompletedStep === previousStep) {
      return {
        title: getStepTitle(thisStep, meta),
        status: 'cancelled'
      };
    } else if (status === undefined || lastCompletedStep < previousStep) {
      return {
        title: getStepTitle(thisStep, meta),
        status: 'incomplete'
      };
    } else if (lastCompletedStep === previousStep) {
      return {
        title: getStepTitle(thisStep, meta, true),
        status: 'inProgress'
      };
    } else {
      return {
        title: getStepTitle(thisStep, meta),
        status: 'complete'
      };
    }
  };

  // The reindexing step is special because it generally lasts longer and can be cancelled mid-flight
  const reindexingDocsStep = {
    title: /*#__PURE__*/_react.default.createElement(ReindexingDocumentsStepTitle, props)
  };
  if (status === _types.ReindexStatus.failed && (lastCompletedStep === _types.ReindexStep.newIndexCreated || lastCompletedStep === _types.ReindexStep.reindexStarted)) {
    reindexingDocsStep.status = 'failed';
    reindexingDocsStep.children = /*#__PURE__*/_react.default.createElement(ErrorCallout, {
      errorMessage
    });
  } else if (status === _types.ReindexStatus.paused && (lastCompletedStep === _types.ReindexStep.newIndexCreated || lastCompletedStep === _types.ReindexStep.reindexStarted)) {
    reindexingDocsStep.status = 'paused';
    reindexingDocsStep.children = /*#__PURE__*/_react.default.createElement(PausedCallout, null);
  } else if (status === _types.ReindexStatus.cancelled && (lastCompletedStep === _types.ReindexStep.newIndexCreated || lastCompletedStep === _types.ReindexStep.reindexStarted)) {
    reindexingDocsStep.status = 'cancelled';
  } else if (status === undefined || lastCompletedStep < _types.ReindexStep.newIndexCreated) {
    reindexingDocsStep.status = 'incomplete';
  } else if (lastCompletedStep === _types.ReindexStep.newIndexCreated || lastCompletedStep === _types.ReindexStep.reindexStarted) {
    reindexingDocsStep.status = 'inProgress';
  } else {
    reindexingDocsStep.status = 'complete';
  }
  const steps = [getProgressStep(_types.ReindexStep.readonly), getProgressStep(_types.ReindexStep.newIndexCreated), reindexingDocsStep, getProgressStep(_types.ReindexStep.aliasCreated), getProgressStep(_types.ReindexStep.originalIndexDeleted)];
  const hasExistingAliases = meta.aliases.length > 0;
  if (hasExistingAliases) {
    steps.push(getProgressStep(_types.ReindexStep.existingAliasesUpdated));
  }
  return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiTitle, {
    size: "xs",
    "data-test-subj": "reindexChecklistTitle"
  }, /*#__PURE__*/_react.default.createElement("h3", null, status === _types.ReindexStatus.inProgress ? /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingInProgressTitle",
    defaultMessage: "Reindexing in progress\u2026 {percents}",
    values: {
      percents: (0, _utils.getReindexProgressLabel)(reindexTaskPercComplete, lastCompletedStep, hasExistingAliases)
    }
  }) : /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "xpack.upgradeAssistant.checkupTab.reindexing.flyout.checklistStep.reindexingChecklistTitle",
    defaultMessage: "Reindexing process"
  }))), /*#__PURE__*/_react.default.createElement(_step_progress.StepProgress, {
    steps: steps
  }));
};
exports.ReindexProgress = ReindexProgress;