"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRuleActionErrors = getRuleActionErrors;
exports.getRuleErrors = getRuleErrors;
exports.hasObjectErrors = void 0;
exports.isValidRule = isValidRule;
exports.validateBaseProperties = validateBaseProperties;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/alerting-plugin/common");
var _parse_duration = require("@kbn/alerting-plugin/common/parse_duration");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validateBaseProperties(ruleObject, config) {
  const validationResult = {
    errors: {}
  };
  const errors = {
    name: new Array(),
    'schedule.interval': new Array(),
    ruleTypeId: new Array(),
    actionConnectors: new Array()
  };
  validationResult.errors = errors;
  if (!ruleObject.name) {
    errors.name.push(_i18n.i18n.translate('xpack.triggersActionsUI.sections.ruleForm.error.requiredNameText', {
      defaultMessage: 'Name is required.'
    }));
  }
  if (ruleObject.schedule.interval.length < 2) {
    errors['schedule.interval'].push(_i18n.i18n.translate('xpack.triggersActionsUI.sections.ruleForm.error.requiredIntervalText', {
      defaultMessage: 'Check interval is required.'
    }));
  } else if (config.minimumScheduleInterval && config.minimumScheduleInterval.enforce) {
    const duration = (0, _parse_duration.parseDuration)(ruleObject.schedule.interval);
    const minimumDuration = (0, _parse_duration.parseDuration)(config.minimumScheduleInterval.value);
    if (duration < minimumDuration) {
      errors['schedule.interval'].push(_i18n.i18n.translate('xpack.triggersActionsUI.sections.ruleForm.error.belowMinimumText', {
        defaultMessage: 'Interval must be at least {minimum}.',
        values: {
          minimum: (0, _parse_duration.formatDuration)(config.minimumScheduleInterval.value, true)
        }
      }));
    }
  }
  const invalidThrottleActions = ruleObject.actions.filter(a => {
    var _a$frequency, _a$frequency2;
    if (!((_a$frequency = a.frequency) !== null && _a$frequency !== void 0 && _a$frequency.throttle)) return false;
    const throttleDuration = (0, _parse_duration.parseDuration)(a.frequency.throttle);
    const intervalDuration = ruleObject.schedule.interval && ruleObject.schedule.interval.length > 1 ? (0, _parse_duration.parseDuration)(ruleObject.schedule.interval) : 0;
    return ((_a$frequency2 = a.frequency) === null || _a$frequency2 === void 0 ? void 0 : _a$frequency2.notifyWhen) === _common.RuleNotifyWhen.THROTTLE && throttleDuration < intervalDuration;
  });
  if (invalidThrottleActions.length) {
    errors['schedule.interval'].push(_i18n.i18n.translate('xpack.triggersActionsUI.sections.ruleForm.error.actionThrottleBelowSchedule', {
      defaultMessage: "Custom action intervals cannot be shorter than the rule's check interval"
    }));
  }
  if (!ruleObject.ruleTypeId) {
    errors.ruleTypeId.push(_i18n.i18n.translate('xpack.triggersActionsUI.sections.ruleForm.error.requiredRuleTypeIdText', {
      defaultMessage: 'Rule type is required.'
    }));
  }
  const emptyConnectorActions = ruleObject.actions.find(actionItem => /^\d+$/.test(actionItem.id) && Object.keys(actionItem.params).length > 0);
  if (emptyConnectorActions !== undefined) {
    errors.actionConnectors.push(_i18n.i18n.translate('xpack.triggersActionsUI.sections.ruleForm.error.requiredActionConnector', {
      defaultMessage: 'Action for {actionTypeId} connector is required.',
      values: {
        actionTypeId: emptyConnectorActions.actionTypeId
      }
    }));
  }
  return validationResult;
}
function getRuleErrors(rule, ruleTypeModel, config) {
  const ruleParamsErrors = ruleTypeModel ? ruleTypeModel.validate(rule.params).errors : [];
  const ruleBaseErrors = validateBaseProperties(rule, config).errors;
  const ruleErrors = {
    ...ruleParamsErrors,
    ...ruleBaseErrors
  };
  return {
    ruleParamsErrors,
    ruleBaseErrors,
    ruleErrors
  };
}
async function getRuleActionErrors(actions, actionTypeRegistry) {
  return await Promise.all(actions.map(async ruleAction => {
    var _actionTypeRegistry$g;
    return (await ((_actionTypeRegistry$g = actionTypeRegistry.get(ruleAction.actionTypeId)) === null || _actionTypeRegistry$g === void 0 ? void 0 : _actionTypeRegistry$g.validateParams(ruleAction.params))).errors;
  }));
}
const hasObjectErrors = errors => !!Object.values(errors).find(errorList => {
  if ((0, _lodash.isObject)(errorList)) return hasObjectErrors(errorList);
  return errorList.length >= 1;
});
exports.hasObjectErrors = hasObjectErrors;
function isValidRule(ruleObject, validationResult, actionsErrors) {
  return !hasObjectErrors(validationResult) && actionsErrors.every(error => !hasObjectErrors(error));
}