"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDeleteTransforms = exports.useDeleteIndexAndTargetIndex = void 0;
var _react = _interopRequireWildcard(require("react"));
var _i18n = require("@kbn/i18n");
var _public = require("@kbn/kibana-react-plugin/public");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _type_guards = require("../../../common/api_schemas/type_guards");
var _errors = require("../../../common/utils/errors");
var _app_dependencies = require("../app_dependencies");
var _common = require("../common");
var _components = require("../components");
var _use_api = require("./use_api");
var _es_index_service = require("../services/es_index_service");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useDeleteIndexAndTargetIndex = items => {
  const {
    http,
    data: {
      dataViews: dataViewsContract
    },
    application: {
      capabilities
    }
  } = (0, _app_dependencies.useAppDependencies)();
  const toastNotifications = (0, _app_dependencies.useToastNotifications)();
  const [deleteDestIndex, setDeleteDestIndex] = (0, _react.useState)(true);
  const [deleteDataView, setDeleteDataView] = (0, _react.useState)(true);
  const [userCanDeleteIndex, setUserCanDeleteIndex] = (0, _react.useState)(false);
  const [dataViewExists, setDataViewExists] = (0, _react.useState)(false);
  const [userCanDeleteDataView, setUserCanDeleteDataView] = (0, _react.useState)(false);
  const toggleDeleteIndex = (0, _react.useCallback)(() => setDeleteDestIndex(!deleteDestIndex), [deleteDestIndex]);
  const toggleDeleteDataView = (0, _react.useCallback)(() => setDeleteDataView(!deleteDataView), [deleteDataView]);
  const checkDataViewExists = (0, _react.useCallback)(async indexName => {
    try {
      const dvExists = await _es_index_service.indexService.dataViewExists(dataViewsContract, indexName);
      setDataViewExists(dvExists);
    } catch (e) {
      const error = (0, _mlErrorUtils.extractErrorMessage)(e);
      toastNotifications.addDanger(_i18n.i18n.translate('xpack.transform.deleteTransform.errorWithCheckingIfDataViewExistsNotificationErrorMessage', {
        defaultMessage: 'An error occurred checking if data view {dataView} exists: {error}',
        values: {
          dataView: indexName,
          error
        }
      }));
    }
  }, [dataViewsContract, toastNotifications]);
  const checkUserIndexPermission = (0, _react.useCallback)(async () => {
    try {
      const userCanDelete = await _es_index_service.indexService.canDeleteIndex(http);
      if (userCanDelete) {
        setUserCanDeleteIndex(true);
      }
      const canDeleteDataView = capabilities.savedObjectsManagement.delete === true || capabilities.indexPatterns.save === true;
      setUserCanDeleteDataView(canDeleteDataView);
      if (canDeleteDataView === false) {
        setDeleteDataView(false);
      }
    } catch (e) {
      toastNotifications.addDanger(_i18n.i18n.translate('xpack.transform.transformList.errorWithCheckingIfUserCanDeleteIndexNotificationErrorMessage', {
        defaultMessage: 'An error occurred checking if user can delete destination index'
      }));
    }
  }, [http, toastNotifications, capabilities]);
  (0, _react.useEffect)(() => {
    checkUserIndexPermission();

    // if user only deleting one transform
    if (items.length === 1) {
      const config = items[0].config;
      const destinationIndex = Array.isArray(config.dest.index) ? config.dest.index[0] : config.dest.index;
      checkDataViewExists(destinationIndex);
    } else {
      setDataViewExists(true);
    }
  }, [checkDataViewExists, checkUserIndexPermission, items]);
  return {
    userCanDeleteIndex,
    userCanDeleteDataView,
    deleteDestIndex,
    dataViewExists,
    deleteDataView,
    toggleDeleteIndex,
    toggleDeleteDataView
  };
};
exports.useDeleteIndexAndTargetIndex = useDeleteIndexAndTargetIndex;
const useDeleteTransforms = () => {
  const {
    overlays,
    theme
  } = (0, _app_dependencies.useAppDependencies)();
  const toastNotifications = (0, _app_dependencies.useToastNotifications)();
  const api = (0, _use_api.useApi)();
  return async reqBody => {
    const results = await api.deleteTransforms(reqBody);
    if (!(0, _type_guards.isDeleteTransformsResponseSchema)(results)) {
      toastNotifications.addDanger({
        title: _i18n.i18n.translate('xpack.transform.transformList.deleteTransformGenericErrorMessage', {
          defaultMessage: 'An error occurred calling the API endpoint to delete transforms.'
        }),
        text: (0, _public.toMountPoint)( /*#__PURE__*/_react.default.createElement(_components.ToastNotificationText, {
          previewTextLength: 50,
          overlays: overlays,
          theme: theme,
          text: (0, _errors.getErrorMessage)(results)
        }), {
          theme$: theme.theme$
        })
      });
      return;
    }
    const isBulk = Object.keys(results).length > 1;
    const successCount = {
      transformDeleted: 0,
      destIndexDeleted: 0,
      destDataViewDeleted: 0
    };
    for (const transformId in results) {
      // hasOwnProperty check to ensure only properties on object itself, and not its prototypes
      if (results.hasOwnProperty(transformId)) {
        var _status$transformDele2, _status$destIndexDele2, _status$destDataViewD2;
        const status = results[transformId];
        const destinationIndex = status.destinationIndex;

        // if we are only deleting one transform, show the success toast messages
        if (!isBulk && status.transformDeleted) {
          var _status$transformDele, _status$destIndexDele, _status$destDataViewD;
          if ((_status$transformDele = status.transformDeleted) !== null && _status$transformDele !== void 0 && _status$transformDele.success) {
            toastNotifications.addSuccess(_i18n.i18n.translate('xpack.transform.transformList.deleteTransformSuccessMessage', {
              defaultMessage: 'Request to delete transform {transformId} acknowledged.',
              values: {
                transformId
              }
            }));
          }
          if ((_status$destIndexDele = status.destIndexDeleted) !== null && _status$destIndexDele !== void 0 && _status$destIndexDele.success) {
            toastNotifications.addSuccess(_i18n.i18n.translate('xpack.transform.deleteTransform.deleteAnalyticsWithIndexSuccessMessage', {
              defaultMessage: 'Request to delete destination index {destinationIndex} acknowledged.',
              values: {
                destinationIndex
              }
            }));
          }
          if ((_status$destDataViewD = status.destDataViewDeleted) !== null && _status$destDataViewD !== void 0 && _status$destDataViewD.success) {
            toastNotifications.addSuccess(_i18n.i18n.translate('xpack.transform.deleteTransform.deleteAnalyticsWithDataViewSuccessMessage', {
              defaultMessage: 'Request to delete data view {destinationIndex} acknowledged.',
              values: {
                destinationIndex
              }
            }));
          }
        } else {
          Object.keys(successCount).forEach(key => {
            var _status$key;
            if ((_status$key = status[key]) !== null && _status$key !== void 0 && _status$key.success) {
              successCount[key] = successCount[key] + 1;
            }
          });
        }
        if ((_status$transformDele2 = status.transformDeleted) !== null && _status$transformDele2 !== void 0 && _status$transformDele2.error) {
          const error = status.transformDeleted.error.reason;
          toastNotifications.addDanger({
            title: _i18n.i18n.translate('xpack.transform.transformList.deleteTransformErrorMessage', {
              defaultMessage: 'An error occurred deleting the transform {transformId}',
              values: {
                transformId
              }
            }),
            text: (0, _public.toMountPoint)( /*#__PURE__*/_react.default.createElement(_components.ToastNotificationText, {
              previewTextLength: 50,
              overlays: overlays,
              theme: theme,
              text: error
            }), {
              theme$: theme.theme$
            })
          });
        }
        if ((_status$destIndexDele2 = status.destIndexDeleted) !== null && _status$destIndexDele2 !== void 0 && _status$destIndexDele2.error) {
          const error = status.destIndexDeleted.error.reason;
          toastNotifications.addDanger({
            title: _i18n.i18n.translate('xpack.transform.deleteTransform.deleteAnalyticsWithIndexErrorMessage', {
              defaultMessage: 'An error occurred deleting destination index {destinationIndex}',
              values: {
                destinationIndex
              }
            }),
            text: (0, _public.toMountPoint)( /*#__PURE__*/_react.default.createElement(_components.ToastNotificationText, {
              previewTextLength: 50,
              overlays: overlays,
              theme: theme,
              text: error
            }), {
              theme$: theme.theme$
            })
          });
        }
        if ((_status$destDataViewD2 = status.destDataViewDeleted) !== null && _status$destDataViewD2 !== void 0 && _status$destDataViewD2.error) {
          const error = status.destDataViewDeleted.error.reason;
          toastNotifications.addDanger({
            title: _i18n.i18n.translate('xpack.transform.deleteTransform.deleteAnalyticsWithDataViewErrorMessage', {
              defaultMessage: 'An error occurred deleting data view {destinationIndex}',
              values: {
                destinationIndex
              }
            }),
            text: (0, _public.toMountPoint)( /*#__PURE__*/_react.default.createElement(_components.ToastNotificationText, {
              previewTextLength: 50,
              overlays: overlays,
              theme: theme,
              text: error
            }), {
              theme$: theme.theme$
            })
          });
        }
      }
    }

    // if we are deleting multiple transforms, combine the success messages
    if (isBulk) {
      if (successCount.transformDeleted > 0) {
        toastNotifications.addSuccess(_i18n.i18n.translate('xpack.transform.transformList.bulkDeleteTransformSuccessMessage', {
          defaultMessage: 'Successfully deleted {count} {count, plural, one {transform} other {transforms}}.',
          values: {
            count: successCount.transformDeleted
          }
        }));
      }
      if (successCount.destIndexDeleted > 0) {
        toastNotifications.addSuccess(_i18n.i18n.translate('xpack.transform.transformList.bulkDeleteDestIndexSuccessMessage', {
          defaultMessage: 'Successfully deleted {count} destination {count, plural, one {index} other {indices}}.',
          values: {
            count: successCount.destIndexDeleted
          }
        }));
      }
      if (successCount.destDataViewDeleted > 0) {
        toastNotifications.addSuccess(_i18n.i18n.translate('xpack.transform.transformList.bulkDeleteDestDataViewSuccessMessage', {
          defaultMessage: 'Successfully deleted {count} destination data {count, plural, one {view} other {views}}.',
          values: {
            count: successCount.destDataViewDeleted
          }
        }));
      }
    }
    _common.refreshTransformList$.next(_common.REFRESH_TRANSFORM_LIST_STATE.REFRESH);
  };
};
exports.useDeleteTransforms = useDeleteTransforms;