"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.syncEditedMonitorBulk = exports.rollbackFailedUpdates = exports.rollbackCompletely = void 0;
var _saved_objects = require("../../../../common/types/saved_objects");
var _runtime_types = require("../../../../common/runtime_types");
var _monitor_upgrade_sender = require("../../telemetry/monitor_upgrade_sender");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateConfigSavedObjects = async ({
  routeContext,
  monitorsToUpdate
}) => {
  return await routeContext.savedObjectsClient.bulkUpdate(monitorsToUpdate.map(({
    previousMonitor,
    monitorWithRevision
  }) => ({
    type: _saved_objects.syntheticsMonitorType,
    id: previousMonitor.id,
    attributes: {
      ...monitorWithRevision,
      [_runtime_types.ConfigKey.CONFIG_ID]: previousMonitor.id,
      [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: monitorWithRevision[_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID] || previousMonitor.id
    }
  })));
};
async function syncUpdatedMonitors({
  spaceId,
  privateLocations,
  routeContext,
  monitorsToUpdate
}) {
  const {
    syntheticsMonitorClient
  } = routeContext;
  return await syntheticsMonitorClient.editMonitors(monitorsToUpdate.map(({
    normalizedMonitor,
    previousMonitor,
    decryptedPreviousMonitor
  }) => ({
    monitor: {
      ...normalizedMonitor,
      [_runtime_types.ConfigKey.CONFIG_ID]: previousMonitor.id,
      [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: normalizedMonitor[_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID] || previousMonitor.id
    },
    id: previousMonitor.id,
    previousMonitor,
    decryptedPreviousMonitor
  })), routeContext, privateLocations, spaceId);
}
const syncEditedMonitorBulk = async ({
  routeContext,
  spaceId,
  monitorsToUpdate,
  privateLocations
}) => {
  const {
    server
  } = routeContext;
  try {
    const [editedMonitorSavedObjects, editSyncResponse] = await Promise.all([updateConfigSavedObjects({
      monitorsToUpdate,
      routeContext
    }), syncUpdatedMonitors({
      monitorsToUpdate,
      routeContext,
      spaceId,
      privateLocations
    })]);
    const {
      failedPolicyUpdates,
      publicSyncErrors
    } = editSyncResponse;
    monitorsToUpdate.forEach(({
      normalizedMonitor,
      previousMonitor
    }) => {
      const editedMonitorSavedObject = editedMonitorSavedObjects === null || editedMonitorSavedObjects === void 0 ? void 0 : editedMonitorSavedObjects.saved_objects.find(obj => obj.id === previousMonitor.id);
      (0, _monitor_upgrade_sender.sendTelemetryEvents)(server.logger, server.telemetry, (0, _monitor_upgrade_sender.formatTelemetryUpdateEvent)(editedMonitorSavedObject, previousMonitor, server.stackVersion, Boolean(normalizedMonitor[_runtime_types.ConfigKey.SOURCE_INLINE]), publicSyncErrors));
    });
    const failedConfigs = await rollbackFailedUpdates({
      monitorsToUpdate,
      routeContext,
      failedPolicyUpdates
    });
    return {
      failedConfigs,
      errors: publicSyncErrors,
      editedMonitors: editedMonitorSavedObjects === null || editedMonitorSavedObjects === void 0 ? void 0 : editedMonitorSavedObjects.saved_objects
    };
  } catch (e) {
    server.logger.error(`Unable to update Synthetics monitors, ${e.message}`);
    await rollbackCompletely({
      routeContext,
      monitorsToUpdate
    });
    throw e;
  }
};
exports.syncEditedMonitorBulk = syncEditedMonitorBulk;
const rollbackCompletely = async ({
  routeContext,
  monitorsToUpdate
}) => {
  const {
    savedObjectsClient,
    server
  } = routeContext;
  try {
    await savedObjectsClient.bulkUpdate(monitorsToUpdate.map(({
      previousMonitor,
      decryptedPreviousMonitor
    }) => ({
      type: _saved_objects.syntheticsMonitorType,
      id: previousMonitor.id,
      attributes: decryptedPreviousMonitor.attributes
    })));
  } catch (e) {
    server.logger.error(`Unable to rollback Synthetics monitors edit ${e.message} `);
  }
};
exports.rollbackCompletely = rollbackCompletely;
const rollbackFailedUpdates = async ({
  routeContext,
  failedPolicyUpdates,
  monitorsToUpdate
}) => {
  if (!failedPolicyUpdates || failedPolicyUpdates.length === 0) {
    return;
  }
  const {
    server,
    savedObjectsClient
  } = routeContext;
  try {
    const failedConfigs = {};
    failedPolicyUpdates.forEach(({
      config,
      error
    }) => {
      if (config && config[_runtime_types.ConfigKey.CONFIG_ID]) {
        failedConfigs[config[_runtime_types.ConfigKey.CONFIG_ID]] = {
          config,
          error
        };
      }
    });
    const monitorsToRevert = monitorsToUpdate.filter(({
      previousMonitor
    }) => {
      return failedConfigs[previousMonitor.id];
    }).map(({
      previousMonitor,
      decryptedPreviousMonitor
    }) => ({
      type: _saved_objects.syntheticsMonitorType,
      id: previousMonitor.id,
      attributes: decryptedPreviousMonitor.attributes
    }));
    if (monitorsToRevert.length > 0) {
      await savedObjectsClient.bulkUpdate(monitorsToRevert);
    }
    return failedConfigs;
  } catch (e) {
    server.logger.error(`Unable to rollback Synthetics monitor failed updates, ${e.message} `);
  }
};
exports.rollbackFailedUpdates = rollbackFailedUpdates;