"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.syncNewMonitor = exports.hydrateMonitorFields = exports.getPrivateLocations = exports.getMonitorNamespace = exports.deleteMonitorIfCreated = exports.createNewSavedObjectMonitor = exports.addSyntheticsMonitorRoute = void 0;
var _uuid = require("uuid");
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _common = require("@kbn/fleet-plugin/common");
var _common2 = require("@kbn/spaces-plugin/common");
var _default_alert_service = require("../default_alerts/default_alert_service");
var _test_now_monitor = require("../synthetics_service/test_now_monitor");
var _saved_objects = require("../../../common/types/saved_objects");
var _private_formatters = require("../../synthetics_service/formatters/private_formatters");
var _private_locations = require("../../legacy_uptime/lib/saved_objects/private_locations");
var _runtime_types = require("../../../common/runtime_types");
var _constants = require("../../../common/constants");
var _monitor_defaults = require("../../../common/constants/monitor_defaults");
var _monitor_validation = require("./monitor_validation");
var _monitor_upgrade_sender = require("../telemetry/monitor_upgrade_sender");
var _secrets = require("../../synthetics_service/utils/secrets");
var _delete_monitor = require("./delete_monitor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const addSyntheticsMonitorRoute = () => ({
  method: 'POST',
  path: _constants.SYNTHETICS_API_URLS.SYNTHETICS_MONITORS,
  validate: {
    body: _configSchema.schema.any(),
    query: _configSchema.schema.object({
      id: _configSchema.schema.maybe(_configSchema.schema.string()),
      preserve_namespace: _configSchema.schema.maybe(_configSchema.schema.boolean()),
      gettingStarted: _configSchema.schema.maybe(_configSchema.schema.boolean())
    })
  },
  writeAccess: true,
  handler: async routeContext => {
    const {
      request,
      response,
      savedObjectsClient,
      server
    } = routeContext;
    // usually id is auto generated, but this is useful for testing
    const {
      id
    } = request.query;
    const monitor = request.body;
    const monitorType = monitor[_runtime_types.ConfigKey.MONITOR_TYPE];
    const monitorWithDefaults = {
      ..._monitor_defaults.DEFAULT_FIELDS[monitorType],
      ...monitor
    };
    const validationResult = (0, _monitor_validation.validateMonitor)(monitorWithDefaults);
    if (!validationResult.valid || !validationResult.decodedMonitor) {
      const {
        reason: message,
        details,
        payload
      } = validationResult;
      return response.badRequest({
        body: {
          message,
          attributes: {
            details,
            ...payload
          }
        }
      });
    }
    const normalizedMonitor = validationResult.decodedMonitor;
    const privateLocations = await getPrivateLocations(savedObjectsClient, normalizedMonitor);
    try {
      const {
        errors,
        newMonitor
      } = await syncNewMonitor({
        normalizedMonitor,
        routeContext,
        id,
        privateLocations
      });
      if (errors && errors.length > 0) {
        return response.ok({
          body: {
            message: 'error pushing monitor to the service',
            attributes: {
              errors
            },
            id: newMonitor.id
          }
        });
      }
      initDefaultAlerts(newMonitor.attributes.name, routeContext);
      setupGettingStarted(newMonitor.id, routeContext);
      return response.ok({
        body: newMonitor
      });
    } catch (getErr) {
      server.logger.error(getErr);
      if (_server.SavedObjectsErrorHelpers.isForbiddenError(getErr)) {
        return response.forbidden({
          body: getErr
        });
      }
      return response.customError({
        body: {
          message: getErr.message
        },
        statusCode: 500
      });
    }
  }
});
exports.addSyntheticsMonitorRoute = addSyntheticsMonitorRoute;
const createNewSavedObjectMonitor = async ({
  id,
  savedObjectsClient,
  normalizedMonitor
}) => {
  return await savedObjectsClient.create(_saved_objects.syntheticsMonitorType, (0, _secrets.formatSecrets)({
    ...normalizedMonitor,
    [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: normalizedMonitor[_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID] || id,
    [_runtime_types.ConfigKey.CONFIG_ID]: id,
    revision: 1
  }), id ? {
    id,
    overwrite: true
  } : undefined);
};
exports.createNewSavedObjectMonitor = createNewSavedObjectMonitor;
const hydrateMonitorFields = ({
  newMonitorId,
  normalizedMonitor,
  routeContext
}) => {
  const {
    server,
    request
  } = routeContext;
  const {
    preserve_namespace: preserveNamespace
  } = request.query;
  return {
    ...normalizedMonitor,
    [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: normalizedMonitor[_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID] || newMonitorId,
    [_runtime_types.ConfigKey.CONFIG_ID]: newMonitorId,
    [_runtime_types.ConfigKey.NAMESPACE]: preserveNamespace ? normalizedMonitor[_runtime_types.ConfigKey.NAMESPACE] : getMonitorNamespace(server, request, normalizedMonitor[_runtime_types.ConfigKey.NAMESPACE])
  };
};
exports.hydrateMonitorFields = hydrateMonitorFields;
const syncNewMonitor = async ({
  id,
  normalizedMonitor,
  privateLocations,
  routeContext
}) => {
  const {
    savedObjectsClient,
    server,
    syntheticsMonitorClient,
    request,
    spaceId
  } = routeContext;
  const newMonitorId = id !== null && id !== void 0 ? id : (0, _uuid.v4)();
  let monitorSavedObject = null;
  const monitorWithNamespace = hydrateMonitorFields({
    normalizedMonitor,
    routeContext,
    newMonitorId
  });
  try {
    var _packagePolicyResult$, _packagePolicyResult$2;
    const newMonitorPromise = createNewSavedObjectMonitor({
      normalizedMonitor: monitorWithNamespace,
      id: newMonitorId,
      savedObjectsClient
    });
    const syncErrorsPromise = syntheticsMonitorClient.addMonitors([{
      monitor: monitorWithNamespace,
      id: newMonitorId
    }], request, savedObjectsClient, privateLocations, spaceId);
    const [monitorSavedObjectN, [packagePolicyResult, syncErrors]] = await Promise.all([newMonitorPromise, syncErrorsPromise]).catch(e => {
      server.logger.error(e);
      throw e;
    });
    if (packagePolicyResult && ((_packagePolicyResult$ = packagePolicyResult === null || packagePolicyResult === void 0 ? void 0 : (_packagePolicyResult$2 = packagePolicyResult.failed) === null || _packagePolicyResult$2 === void 0 ? void 0 : _packagePolicyResult$2.length) !== null && _packagePolicyResult$ !== void 0 ? _packagePolicyResult$ : []) > 0) {
      const failed = packagePolicyResult.failed.map(f => f.error);
      throw new Error(failed.join(', '));
    }
    monitorSavedObject = monitorSavedObjectN;
    (0, _monitor_upgrade_sender.sendTelemetryEvents)(server.logger, server.telemetry, (0, _monitor_upgrade_sender.formatTelemetryEvent)({
      errors: syncErrors,
      monitor: monitorSavedObject,
      isInlineScript: Boolean(normalizedMonitor[_runtime_types.ConfigKey.SOURCE_INLINE]),
      stackVersion: server.stackVersion
    }));
    return {
      errors: syncErrors,
      newMonitor: monitorSavedObject
    };
  } catch (e) {
    server.logger.error(`Unable to create Synthetics monitor ${monitorWithNamespace[_runtime_types.ConfigKey.NAME]}`);
    await deleteMonitorIfCreated({
      newMonitorId,
      routeContext
    });
    server.logger.error(e);
    throw e;
  }
};
exports.syncNewMonitor = syncNewMonitor;
const deleteMonitorIfCreated = async ({
  newMonitorId,
  routeContext
}) => {
  const {
    server,
    savedObjectsClient
  } = routeContext;
  try {
    const encryptedMonitor = await savedObjectsClient.get(_saved_objects.syntheticsMonitorType, newMonitorId);
    if (encryptedMonitor) {
      await savedObjectsClient.delete(_saved_objects.syntheticsMonitorType, newMonitorId);
      await (0, _delete_monitor.deleteMonitor)({
        routeContext,
        monitorId: newMonitorId
      });
    }
  } catch (e) {
    // ignore errors here
    server.logger.error(e);
  }
};
exports.deleteMonitorIfCreated = deleteMonitorIfCreated;
const getPrivateLocations = async (soClient, normalizedMonitor) => {
  const {
    locations
  } = normalizedMonitor;
  const hasPrivateLocation = locations.filter(location => !location.isServiceManaged);
  if (hasPrivateLocation.length === 0) {
    return [];
  }
  return await (0, _private_locations.getSyntheticsPrivateLocations)(soClient);
};
exports.getPrivateLocations = getPrivateLocations;
const getMonitorNamespace = (server, request, configuredNamespace) => {
  var _server$spaces$spaces, _server$spaces;
  const spaceId = (_server$spaces$spaces = (_server$spaces = server.spaces) === null || _server$spaces === void 0 ? void 0 : _server$spaces.spacesService.getSpaceId(request)) !== null && _server$spaces$spaces !== void 0 ? _server$spaces$spaces : _common2.DEFAULT_SPACE_ID;
  const kibanaNamespace = (0, _private_formatters.formatKibanaNamespace)(spaceId);
  const namespace = configuredNamespace === _monitor_defaults.DEFAULT_NAMESPACE_STRING ? kibanaNamespace : configuredNamespace;
  const {
    error
  } = (0, _common.isValidNamespace)(namespace);
  if (error) {
    throw new Error(`Cannot save monitor. Monitor namespace is invalid: ${error}`);
  }
  return namespace;
};
exports.getMonitorNamespace = getMonitorNamespace;
const initDefaultAlerts = (name, routeContext) => {
  const {
    server,
    savedObjectsClient,
    context
  } = routeContext;
  try {
    // we do this async, so we don't block the user, error handling will be done on the UI via separate api
    const defaultAlertService = new _default_alert_service.DefaultAlertService(context, server, savedObjectsClient);
    defaultAlertService.setupDefaultAlerts().then(() => {
      server.logger.debug(`Successfully created default alert for monitor: ${name}`);
    });
  } catch (e) {
    server.logger.error(`Error creating default alert: ${e} for monitor: ${name}`);
  }
};
const setupGettingStarted = (configId, routeContext) => {
  const {
    server,
    request
  } = routeContext;
  try {
    const {
      gettingStarted
    } = request.query;
    if (gettingStarted) {
      // ignore await, since we don't want to block the response
      (0, _test_now_monitor.triggerTestNow)(configId, routeContext).then(() => {
        server.logger.debug(`Successfully triggered test for monitor: ${configId}`);
      }).catch(e => {
        server.logger.error(`Error triggering test for monitor: ${configId}: ${e}`);
      });
    }
  } catch (e) {
    server.logger.info(`Error triggering test for getting started monitor: ${configId}`);
    server.logger.error(e);
  }
};