"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.migration880 = void 0;
var _lodash = require("lodash");
var _monitor_management = require("../../../../../../common/constants/monitor_management");
var _runtime_types = require("../../../../../../common/runtime_types");
var _monitor_defaults = require("../../../../../../common/constants/monitor_defaults");
var _synthetics_monitor = require("../../synthetics_monitor");
var _monitor_validation = require("../../../../../routes/monitor_cruds/monitor_validation");
var _secrets = require("../../../../../synthetics_service/utils/secrets");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const migration880 = encryptedSavedObjects => {
  return encryptedSavedObjects.createMigration({
    isMigrationNeededPredicate: function shouldBeMigrated(doc) {
      return true;
    },
    migration: (doc, logger) => {
      let migrated = doc;
      migrated = {
        ...migrated,
        attributes: {
          ...migrated.attributes,
          [_runtime_types.ConfigKey.SCHEDULE]: {
            number: getNearestSupportedSchedule(migrated.attributes[_runtime_types.ConfigKey.SCHEDULE].number),
            unit: _runtime_types.ScheduleUnit.MINUTES
          },
          [_runtime_types.ConfigKey.ALERT_CONFIG]: migrated.attributes[_runtime_types.ConfigKey.ALERT_CONFIG] || {
            status: {
              enabled: true
            }
          },
          // when any action to change a project monitor configuration is taken
          // outside of the synthetics agent cli, we should set the config hash back
          // to an empty string so that the project monitors configuration
          // will be updated on next push
          [_runtime_types.ConfigKey.CONFIG_HASH]: ''
        }
      };
      if (migrated.attributes.type === 'browser') {
        migrated = updateThrottlingFields(migrated, logger);
        try {
          const normalizedMonitorAttributes = (0, _secrets.normalizeMonitorSecretAttributes)(migrated.attributes);
          migrated = {
            ...migrated,
            attributes: omitZipUrlFields(normalizedMonitorAttributes)
          };
        } catch (e) {
          logger.log.warn(`Failed to remove ZIP URL fields from legacy Synthetics monitor: ${e.message}`);
          return migrated;
        }
      }
      return migrated;
    },
    inputType: _synthetics_monitor.LEGACY_SYNTHETICS_MONITOR_ENCRYPTED_TYPE,
    migratedType: _synthetics_monitor.SYNTHETICS_MONITOR_ENCRYPTED_TYPE
  });
};
exports.migration880 = migration880;
const getNearestSupportedSchedule = currentSchedule => {
  try {
    const closest = _monitor_defaults.ALLOWED_SCHEDULES_IN_MINUTES.reduce(function (prev, curr) {
      const supportedSchedule = parseFloat(curr);
      const currSchedule = parseFloat(currentSchedule);
      const prevSupportedSchedule = parseFloat(prev);
      return Math.abs(supportedSchedule - currSchedule) < Math.abs(prevSupportedSchedule - currSchedule) ? curr : prev;
    });
    return closest;
  } catch {
    return '10';
  }
};
const omitZipUrlFields = fields => {
  const metadata = fields[_runtime_types.ConfigKey.METADATA];
  const updatedMetadata = (0, _lodash.omit)(metadata || {}, 'is_zip_url_tls_enabled');
  // will return only fields that match the current type defs, which omit
  // zip url fields

  const validationResult = (0, _monitor_validation.validateMonitor)({
    ...fields,
    [_runtime_types.ConfigKey.METADATA]: updatedMetadata
  });
  if (!validationResult.valid || !validationResult.decodedMonitor) {
    throw new Error(`Monitor is not valid: ${validationResult.reason}. ${validationResult.details}`);
  }
  return (0, _secrets.formatSecrets)(validationResult.decodedMonitor);
};
const updateThrottlingFields = (doc, logger) => {
  try {
    const {
      attributes
    } = doc;
    const migrated = {
      ...doc,
      attributes: {
        ...doc.attributes,
        [_runtime_types.ConfigKey.CONFIG_HASH]: ''
      }
    };
    const isThrottlingEnabled = attributes[_monitor_management.LegacyConfigKey.IS_THROTTLING_ENABLED];
    if (isThrottlingEnabled) {
      const defaultProfileValue = _monitor_defaults.PROFILES_MAP[_monitor_defaults.PROFILE_VALUES_ENUM.DEFAULT].value;
      const download = String(attributes[_monitor_management.LegacyConfigKey.DOWNLOAD_SPEED]) || defaultProfileValue.download;
      const upload = String(attributes[_monitor_management.LegacyConfigKey.UPLOAD_SPEED]) || defaultProfileValue.upload;
      const latency = String(attributes[_monitor_management.LegacyConfigKey.LATENCY]) || defaultProfileValue.latency;
      migrated.attributes[_runtime_types.ConfigKey.THROTTLING_CONFIG] = getMatchingThrottlingConfig(download, upload, latency);
    } else {
      migrated.attributes[_runtime_types.ConfigKey.THROTTLING_CONFIG] = _monitor_defaults.PROFILES_MAP[_monitor_defaults.PROFILE_VALUES_ENUM.NO_THROTTLING];
    }

    // filter out legacy throttling fields
    return {
      ...migrated,
      attributes: (0, _lodash.omit)(migrated.attributes, [_monitor_management.LegacyConfigKey.THROTTLING_CONFIG, _monitor_management.LegacyConfigKey.IS_THROTTLING_ENABLED, _monitor_management.LegacyConfigKey.DOWNLOAD_SPEED, _monitor_management.LegacyConfigKey.UPLOAD_SPEED, _monitor_management.LegacyConfigKey.LATENCY])
    };
  } catch (e) {
    logger.log.warn(`Failed to migrate throttling fields from legacy Synthetics monitor: ${e.message}`);
    const {
      attributes
    } = doc;
    attributes[_runtime_types.ConfigKey.THROTTLING_CONFIG] = _monitor_defaults.PROFILES_MAP[_monitor_defaults.PROFILE_VALUES_ENUM.DEFAULT];
    return {
      ...doc,
      attributes: (0, _lodash.omit)(attributes, [_monitor_management.LegacyConfigKey.THROTTLING_CONFIG, _monitor_management.LegacyConfigKey.IS_THROTTLING_ENABLED, _monitor_management.LegacyConfigKey.DOWNLOAD_SPEED, _monitor_management.LegacyConfigKey.UPLOAD_SPEED, _monitor_management.LegacyConfigKey.LATENCY])
    };
  }
};
const getMatchingThrottlingConfig = (download, upload, latency) => {
  const matchedProfile = _monitor_defaults.PROFILE_VALUES.find(({
    value
  }) => {
    return Number(value === null || value === void 0 ? void 0 : value.download) === Number(download) && Number(value === null || value === void 0 ? void 0 : value.upload) === Number(upload) && Number(value === null || value === void 0 ? void 0 : value.latency) === Number(latency);
  });
  if (matchedProfile) {
    return matchedProfile;
  } else {
    return {
      id: _monitor_defaults.PROFILE_VALUES_ENUM.CUSTOM,
      label: _monitor_defaults.CUSTOM_LABEL,
      value: {
        download: String(download),
        upload: String(upload),
        latency: String(latency)
      }
    };
  }
};