"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executor = executor;
var _lib = require("./lib");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function executor({
  previousStartedAt,
  startedAt: windowEnd,
  services,
  params,
  rule,
  state,
  logger
}) {
  const boundariesRequestMeta = {
    geoField: params.geoField,
    boundaryIndexTitle: params.boundaryIndexTitle,
    boundaryGeoField: params.boundaryGeoField,
    boundaryNameField: params.boundaryNameField,
    boundaryIndexQuery: params.boundaryIndexQuery
  };
  const {
    shapesFilters,
    shapesIdsNamesMap
  } = state.shapesFilters && (0, _lib.canSkipBoundariesFetch)(boundariesRequestMeta, state.boundariesRequestMeta) ? state : await (0, _lib.getShapeFilters)(boundariesRequestMeta, services.scopedClusterClient.asCurrentUser);
  let windowStart = previousStartedAt;
  if (!windowStart) {
    logger.debug(`alert ${_constants.GEO_CONTAINMENT_ID}:${rule.id} alert initialized. Collecting data`);
    // Consider making first time window configurable?
    const START_TIME_WINDOW = 1;
    windowStart = new Date(windowEnd);
    windowStart.setMinutes(windowStart.getMinutes() - START_TIME_WINDOW);
  }
  const results = await (0, _lib.executeEsQuery)(params, services.scopedClusterClient.asCurrentUser, shapesFilters, windowStart, windowEnd);
  const currLocationMap = (0, _lib.transformResults)(results, params.dateField, params.geoField);
  const prevLocationMap = new Map([...Object.entries(state.prevLocationMap || {})]);
  const {
    activeEntities,
    inactiveEntities
  } = (0, _lib.getEntitiesAndGenerateAlerts)(prevLocationMap, currLocationMap, services.alertFactory, shapesIdsNamesMap, windowEnd);
  const {
    getRecoveredAlerts
  } = services.alertFactory.done();
  for (const recoveredAlert of getRecoveredAlerts()) {
    const recoveredAlertId = recoveredAlert.getId();
    try {
      const context = (0, _lib.getRecoveredAlertContext)({
        alertId: recoveredAlertId,
        activeEntities,
        inactiveEntities,
        windowEnd
      });
      if (context) {
        recoveredAlert.setContext(context);
      }
    } catch (e) {
      logger.warn(`Unable to set alert context for recovered alert, error: ${e.message}`);
    }
  }
  return {
    state: {
      boundariesRequestMeta,
      shapesFilters,
      shapesIdsNamesMap,
      prevLocationMap: Object.fromEntries(activeEntities)
    }
  };
}