"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSideNavSelectedId = exports.useSideNavItems = exports.usePartitionFooterNavItems = void 0;
var _react = require("react");
var _reactRouterDom = require("react-router-dom");
var _fp = require("lodash/fp");
var _common = require("@kbn/security-solution-plugin/common");
var _services = require("../services");
var _use_link_props = require("./use_link_props");
var _use_nav_links = require("./use_nav_links");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isFooterNavItem = id => id === _common.SecurityPageName.landing || id === _common.SecurityPageName.administration;
const isGetStartedNavItem = id => id === _common.SecurityPageName.landing;

// DFS for the sideNavItem matching the current `pathname`, returns all item hierarchy when found
const findItemsByPath = (sideNavItems, pathname) => {
  for (const sideNavItem of sideNavItems) {
    var _sideNavItem$items;
    if ((_sideNavItem$items = sideNavItem.items) !== null && _sideNavItem$items !== void 0 && _sideNavItem$items.length) {
      const found = findItemsByPath(sideNavItem.items, pathname);
      if (found.length) {
        found.unshift(sideNavItem);
        return found;
      }
    }
    if ((0, _reactRouterDom.matchPath)(pathname, {
      path: sideNavItem.href
    })) {
      return [sideNavItem];
    }
  }
  return [];
};

/**
 * Returns all the formatted SideNavItems, including external links
 */
const useSideNavItems = () => {
  const navLinks = (0, _use_nav_links.useNavLinks)();
  const getLinkProps = (0, _use_link_props.useGetLinkProps)();
  const securitySideNavItems = (0, _react.useMemo)(() => navLinks.reduce((items, navLink) => {
    if (navLink.disabled) {
      return items;
    }
    if (isGetStartedNavItem(navLink.id)) {
      items.push({
        id: navLink.id,
        label: navLink.title.toUpperCase(),
        ...getLinkProps({
          deepLinkId: navLink.id
        }),
        labelSize: 'xs',
        iconType: 'launch',
        appendSeparator: true
      });
    } else {
      var _navLink$categories, _navLink$links;
      // default sideNavItem formatting
      items.push({
        id: navLink.id,
        label: navLink.title,
        ...getLinkProps({
          deepLinkId: navLink.id
        }),
        ...(((_navLink$categories = navLink.categories) === null || _navLink$categories === void 0 ? void 0 : _navLink$categories.length) && {
          categories: navLink.categories
        }),
        ...(((_navLink$links = navLink.links) === null || _navLink$links === void 0 ? void 0 : _navLink$links.length) && {
          items: navLink.links.reduce((acc, current) => {
            if (!current.disabled) {
              acc.push({
                id: current.id,
                label: current.title,
                description: current.description,
                isBeta: current.isBeta,
                betaOptions: current.betaOptions,
                ...getLinkProps({
                  deepLinkId: current.id
                })
              });
            }
            return acc;
          }, [])
        })
      });
    }
    return items;
  }, []), [getLinkProps, navLinks]);
  const sideNavItems = useAddExternalSideNavItems(securitySideNavItems);
  return sideNavItems;
};

/**
 * @param securitySideNavItems the sideNavItems for Security pages
 * @returns sideNavItems with Security and external links
 */
exports.useSideNavItems = useSideNavItems;
const useAddExternalSideNavItems = securitySideNavItems => {
  const sideNavItemsWithExternals = (0, _react.useMemo)(() => [...securitySideNavItems
  // TODO: add external links. e.g.:
  // {
  //   id: 'ml',
  //   label: 'Machine Learning Jobs',
  //   ...getLinkProps({ appId: 'ml', path: '/jobs' }),
  //   links: [...]
  // },
  ], [securitySideNavItems]);
  return sideNavItemsWithExternals;
};

/**
 * Partitions the sideNavItems into main and footer SideNavItems
 * @param sideNavItems array for all SideNavItems
 * @returns `[items, footerItems]` to be used in the side navigation component
 */
const usePartitionFooterNavItems = sideNavItems => (0, _react.useMemo)(() => (0, _fp.partition)(item => !isFooterNavItem(item.id), sideNavItems), [sideNavItems]);

/**
 * Returns the selected item id, which is the root item in the links hierarchy
 */
exports.usePartitionFooterNavItems = usePartitionFooterNavItems;
const useSideNavSelectedId = sideNavItems => {
  const {
    http
  } = (0, _services.useKibana)().services;
  const {
    pathname
  } = (0, _reactRouterDom.useLocation)();
  const selectedId = (0, _react.useMemo)(() => {
    var _rootNavItem$id;
    const [rootNavItem] = findItemsByPath(sideNavItems, http.basePath.prepend(pathname));
    return (_rootNavItem$id = rootNavItem === null || rootNavItem === void 0 ? void 0 : rootNavItem.id) !== null && _rootNavItem$id !== void 0 ? _rootNavItem$id : '';
  }, [sideNavItems, pathname, http]);
  return selectedId;
};
exports.useSideNavSelectedId = useSideNavSelectedId;