"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateExceptionListItem = exports.updateExceptionList = exports.getExceptionFilterFromExceptions = exports.getExceptionFilterFromExceptionListIds = exports.fetchExceptionListsItemsByListIds = exports.fetchExceptionLists = exports.fetchExceptionListItemById = exports.fetchExceptionListById = exports.exportExceptionList = exports.duplicateExceptionList = exports.deleteExceptionListItemById = exports.deleteExceptionListById = exports.addExceptionListItem = exports.addExceptionList = exports.addEndpointExceptionList = void 0;
var _TaskEither = require("fp-ts/lib/TaskEither");
var _function = require("fp-ts/lib/function");
var _securitysolutionIoTsUtils = require("@kbn/securitysolution-io-ts-utils");
var _securitysolutionIoTsListTypes = require("@kbn/securitysolution-io-ts-list-types");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _fp_utils = require("../fp_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Add new ExceptionList
 *
 * @param http Kibana http service
 * @param list exception list to add
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 *
 */
const addExceptionList = async ({
  http,
  list,
  signal
}) => http.fetch(_securitysolutionListConstants.EXCEPTION_LIST_URL, {
  body: JSON.stringify(list),
  method: 'POST',
  signal
});
const addExceptionListWithValidation = async ({
  http,
  list,
  signal
}) => (0, _function.flow)(() => (0, _TaskEither.tryCatch)(() => addExceptionList({
  http,
  list,
  signal
}), _fp_utils.toError), (0, _TaskEither.chain)(response => (0, _TaskEither.fromEither)((0, _securitysolutionIoTsUtils.validateEither)(_securitysolutionIoTsListTypes.exceptionListSchema, response))), (0, _function.flow)(_fp_utils.toPromise))();
exports.addExceptionList = addExceptionListWithValidation;
/**
 * Add new ExceptionListItem
 *
 * @param http Kibana http service
 * @param listItem exception list item to add
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 *
 */
const addExceptionListItem = async ({
  http,
  listItem,
  signal
}) => http.fetch(_securitysolutionListConstants.EXCEPTION_LIST_ITEM_URL, {
  body: JSON.stringify(listItem),
  method: 'POST',
  signal
});
const addExceptionListItemWithValidation = async ({
  http,
  listItem,
  signal
}) => (0, _function.flow)(() => (0, _TaskEither.tryCatch)(() => addExceptionListItem({
  http,
  listItem,
  signal
}), _fp_utils.toError), (0, _TaskEither.chain)(response => (0, _TaskEither.fromEither)((0, _securitysolutionIoTsUtils.validateEither)(_securitysolutionIoTsListTypes.exceptionListItemSchema, response))), (0, _function.flow)(_fp_utils.toPromise))();
exports.addExceptionListItem = addExceptionListItemWithValidation;
/**
 * Update existing ExceptionList
 *
 * @param http Kibana http service
 * @param list exception list to add
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 *
 */
const updateExceptionList = async ({
  http,
  list,
  signal
}) => http.fetch(_securitysolutionListConstants.EXCEPTION_LIST_URL, {
  body: JSON.stringify(list),
  method: 'PUT',
  signal
});
const updateExceptionListWithValidation = async ({
  http,
  list,
  signal
}) => (0, _function.flow)(() => (0, _TaskEither.tryCatch)(() => updateExceptionList({
  http,
  list,
  signal
}), _fp_utils.toError), (0, _TaskEither.chain)(response => (0, _TaskEither.fromEither)((0, _securitysolutionIoTsUtils.validateEither)(_securitysolutionIoTsListTypes.exceptionListSchema, response))), (0, _function.flow)(_fp_utils.toPromise))();
exports.updateExceptionList = updateExceptionListWithValidation;
/**
 * Update existing ExceptionListItem
 *
 * @param http Kibana http service
 * @param listItem exception list item to add
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 *
 */
const updateExceptionListItem = async ({
  http,
  listItem,
  signal
}) => http.fetch(_securitysolutionListConstants.EXCEPTION_LIST_ITEM_URL, {
  body: JSON.stringify(listItem),
  method: 'PUT',
  signal
});
const updateExceptionListItemWithValidation = async ({
  http,
  listItem,
  signal
}) => (0, _function.flow)(() => (0, _TaskEither.tryCatch)(() => updateExceptionListItem({
  http,
  listItem,
  signal
}), _fp_utils.toError), (0, _TaskEither.chain)(response => (0, _TaskEither.fromEither)((0, _securitysolutionIoTsUtils.validateEither)(_securitysolutionIoTsListTypes.exceptionListItemSchema, response))), (0, _function.flow)(_fp_utils.toPromise))();
exports.updateExceptionListItem = updateExceptionListItemWithValidation;
/**
 * Fetch all ExceptionLists (optionally by namespaceType)
 *
 * @param http Kibana http service
 * @param namespaceTypes ExceptionList namespace_types of lists to find
 * @param filters search bar filters
 * @param pagination optional
 * @param signal to cancel request
 *
 * @throws An error if request params or response is not OK
 */
const fetchExceptionLists = async ({
  http,
  filters,
  namespaceTypes,
  pagination,
  signal,
  sort
}) => {
  const query = {
    filter: filters || undefined,
    namespace_type: namespaceTypes,
    page: pagination.page ? `${pagination.page}` : '1',
    per_page: pagination.perPage ? `${pagination.perPage}` : '20',
    sort_field: sort !== null && sort !== void 0 && sort.field ? sort === null || sort === void 0 ? void 0 : sort.field : 'exception-list.created_at',
    sort_order: sort !== null && sort !== void 0 && sort.order ? sort === null || sort === void 0 ? void 0 : sort.order : 'desc'
  };
  return http.fetch(`${_securitysolutionListConstants.EXCEPTION_LIST_URL}/_find`, {
    method: 'GET',
    query,
    signal
  });
};
const fetchExceptionListsWithValidation = async ({
  filters,
  http,
  namespaceTypes,
  pagination,
  signal,
  sort
}) => (0, _function.flow)(() => (0, _TaskEither.tryCatch)(() => fetchExceptionLists({
  filters,
  http,
  namespaceTypes,
  pagination,
  signal,
  sort
}), _fp_utils.toError), (0, _TaskEither.chain)(response => (0, _TaskEither.fromEither)((0, _securitysolutionIoTsUtils.validateEither)(_securitysolutionIoTsListTypes.foundExceptionListSchema, response))), (0, _function.flow)(_fp_utils.toPromise))();
exports.fetchExceptionLists = fetchExceptionListsWithValidation;
/**
 * Fetch an ExceptionList by providing a ExceptionList ID
 *
 * @param http Kibana http service
 * @param id ExceptionList ID (not list_id)
 * @param namespaceType ExceptionList namespace_type
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
const fetchExceptionListById = async ({
  http,
  id,
  namespaceType,
  signal
}) => http.fetch(_securitysolutionListConstants.EXCEPTION_LIST_URL, {
  method: 'GET',
  query: {
    id,
    namespace_type: namespaceType
  },
  signal
});
const fetchExceptionListByIdWithValidation = async ({
  http,
  id,
  namespaceType,
  signal
}) => (0, _function.flow)(() => (0, _TaskEither.tryCatch)(() => fetchExceptionListById({
  http,
  id,
  namespaceType,
  signal
}), _fp_utils.toError), (0, _TaskEither.chain)(response => (0, _TaskEither.fromEither)((0, _securitysolutionIoTsUtils.validateEither)(_securitysolutionIoTsListTypes.exceptionListSchema, response))), (0, _function.flow)(_fp_utils.toPromise))();
exports.fetchExceptionListById = fetchExceptionListByIdWithValidation;
/**
 * Fetch an ExceptionList's ExceptionItems by providing a ExceptionList list_id
 *
 * @param http Kibana http service
 * @param listIds ExceptionList list_ids (not ID)
 * @param namespaceTypes ExceptionList namespace_types
 * @param search optional - simple search string
 * @param filter optional
 * @param pagination optional
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
const fetchExceptionListsItemsByListIds = async ({
  http,
  listIds,
  namespaceTypes,
  filter,
  pagination,
  search,
  signal
}) => {
  const query = {
    list_id: listIds.join(','),
    namespace_type: namespaceTypes.join(','),
    page: pagination.page ? `${pagination.page}` : '1',
    per_page: pagination.perPage ? `${pagination.perPage}` : '20',
    search,
    sort_field: 'exception-list.created_at',
    sort_order: 'desc',
    filter
  };
  return http.fetch(`${_securitysolutionListConstants.EXCEPTION_LIST_ITEM_URL}/_find`, {
    method: 'GET',
    query,
    signal
  });
};
const fetchExceptionListsItemsByListIdsWithValidation = async ({
  filter,
  http,
  listIds,
  namespaceTypes,
  pagination,
  search,
  signal
}) => (0, _function.flow)(() => (0, _TaskEither.tryCatch)(() => fetchExceptionListsItemsByListIds({
  filter,
  http,
  listIds,
  namespaceTypes,
  pagination,
  search,
  signal
}), _fp_utils.toError), (0, _TaskEither.chain)(response => (0, _TaskEither.fromEither)((0, _securitysolutionIoTsUtils.validateEither)(_securitysolutionIoTsListTypes.foundExceptionListItemSchema, response))), (0, _function.flow)(_fp_utils.toPromise))();
exports.fetchExceptionListsItemsByListIds = fetchExceptionListsItemsByListIdsWithValidation;
/**
 * Fetch an ExceptionListItem by providing a ExceptionListItem ID
 *
 * @param http Kibana http service
 * @param id ExceptionListItem ID (not item_id)
 * @param namespaceType ExceptionList namespace_type
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
const fetchExceptionListItemById = async ({
  http,
  id,
  namespaceType,
  signal
}) => http.fetch(_securitysolutionListConstants.EXCEPTION_LIST_ITEM_URL, {
  method: 'GET',
  query: {
    id,
    namespace_type: namespaceType
  },
  signal
});
const fetchExceptionListItemByIdWithValidation = async ({
  http,
  id,
  namespaceType,
  signal
}) => (0, _function.flow)(() => (0, _TaskEither.tryCatch)(() => fetchExceptionListItemById({
  http,
  id,
  namespaceType,
  signal
}), _fp_utils.toError), (0, _TaskEither.chain)(response => (0, _TaskEither.fromEither)((0, _securitysolutionIoTsUtils.validateEither)(_securitysolutionIoTsListTypes.exceptionListItemSchema, response))), (0, _function.flow)(_fp_utils.toPromise))();
exports.fetchExceptionListItemById = fetchExceptionListItemByIdWithValidation;
/**
 * Delete an ExceptionList by providing a ExceptionList ID
 *
 * @param http Kibana http service
 * @param id ExceptionList ID (not list_id)
 * @param namespaceType ExceptionList namespace_type
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
const deleteExceptionListById = async ({
  http,
  id,
  namespaceType,
  signal
}) => http.fetch(_securitysolutionListConstants.EXCEPTION_LIST_URL, {
  method: 'DELETE',
  query: {
    id,
    namespace_type: namespaceType
  },
  signal
});
const deleteExceptionListByIdWithValidation = async ({
  http,
  id,
  namespaceType,
  signal
}) => (0, _function.flow)(() => (0, _TaskEither.tryCatch)(() => deleteExceptionListById({
  http,
  id,
  namespaceType,
  signal
}), _fp_utils.toError), (0, _TaskEither.chain)(response => (0, _TaskEither.fromEither)((0, _securitysolutionIoTsUtils.validateEither)(_securitysolutionIoTsListTypes.exceptionListSchema, response))), (0, _function.flow)(_fp_utils.toPromise))();
exports.deleteExceptionListById = deleteExceptionListByIdWithValidation;
/**
 * Delete an ExceptionListItem by providing a ExceptionListItem ID
 *
 * @param http Kibana http service
 * @param id ExceptionListItem ID (not item_id)
 * @param namespaceType ExceptionList namespace_type
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
const deleteExceptionListItemById = async ({
  http,
  id,
  namespaceType,
  signal
}) => http.fetch(_securitysolutionListConstants.EXCEPTION_LIST_ITEM_URL, {
  method: 'DELETE',
  query: {
    id,
    namespace_type: namespaceType
  },
  signal
});
const deleteExceptionListItemByIdWithValidation = async ({
  http,
  id,
  namespaceType,
  signal
}) => (0, _function.flow)(() => (0, _TaskEither.tryCatch)(() => deleteExceptionListItemById({
  http,
  id,
  namespaceType,
  signal
}), _fp_utils.toError), (0, _TaskEither.chain)(response => (0, _TaskEither.fromEither)((0, _securitysolutionIoTsUtils.validateEither)(_securitysolutionIoTsListTypes.exceptionListItemSchema, response))), (0, _function.flow)(_fp_utils.toPromise))();
exports.deleteExceptionListItemById = deleteExceptionListItemByIdWithValidation;
/**
 * Add new Endpoint ExceptionList
 *
 * @param http Kibana http service
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 *
 */
const addEndpointExceptionList = async ({
  http,
  signal
}) => http.fetch(_securitysolutionListConstants.ENDPOINT_LIST_URL, {
  method: 'POST',
  signal
});
const addEndpointExceptionListWithValidation = async ({
  http,
  signal
}) => (0, _function.flow)(() => (0, _TaskEither.tryCatch)(() => addEndpointExceptionList({
  http,
  signal
}), _fp_utils.toError), (0, _TaskEither.chain)(response => (0, _TaskEither.fromEither)((0, _securitysolutionIoTsUtils.validateEither)(_securitysolutionIoTsListTypes.createEndpointListSchema, response))), (0, _function.flow)(_fp_utils.toPromise))();
exports.addEndpointExceptionList = addEndpointExceptionListWithValidation;
/**
 * Export an ExceptionList by providing a ExceptionList ID
 *
 * @param http Kibana http service
 * @param id ExceptionList ID (not list_id)
 * @param includeExpiredExceptions boolean for including expired exceptions
 * @param listId ExceptionList LIST_ID (not id)
 * @param namespaceType ExceptionList namespace_type
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
const exportExceptionList = async ({
  http,
  id,
  includeExpiredExceptions,
  listId,
  namespaceType,
  signal
}) => http.fetch(`${_securitysolutionListConstants.EXCEPTION_LIST_URL}/_export`, {
  method: 'POST',
  query: {
    id,
    list_id: listId,
    namespace_type: namespaceType,
    include_expired_exceptions: includeExpiredExceptions
  },
  signal
});

/**
 * Create a Filter query from an exception list id
 *
 * @param exceptionListId The id of the exception list from which create a Filter query
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
exports.exportExceptionList = exportExceptionList;
const getExceptionFilterFromExceptionListIds = async ({
  alias,
  chunkSize,
  exceptionListIds,
  excludeExceptions,
  http,
  signal
}) => http.fetch(_securitysolutionListConstants.EXCEPTION_FILTER, {
  method: 'POST',
  body: JSON.stringify({
    exception_list_ids: exceptionListIds,
    type: 'exception_list_ids',
    alias,
    exclude_exceptions: excludeExceptions,
    chunk_size: chunkSize
  }),
  signal
});

/**
 * Create a Filter query from a list of exceptions
 *
 * @param exceptions Exception items to be made into a `Filter` query
 * @param signal AbortSignal for cancelling request
 *
 * @throws An error if response is not OK
 */
exports.getExceptionFilterFromExceptionListIds = getExceptionFilterFromExceptionListIds;
const getExceptionFilterFromExceptions = async ({
  exceptions,
  alias,
  excludeExceptions,
  http,
  chunkSize,
  signal
}) => http.fetch(_securitysolutionListConstants.EXCEPTION_FILTER, {
  method: 'POST',
  body: JSON.stringify({
    exceptions,
    type: 'exception_items',
    alias,
    exclude_exceptions: excludeExceptions,
    chunk_size: chunkSize
  }),
  signal
});

/**
 * Duplicate an ExceptionList and its items by providing a ExceptionList list_id
 *
 * @param http Kibana http service
 * @param includeExpiredExceptions boolean for including exception items with expired TTL
 * @param listId ExceptionList LIST_ID (not id)
 * @param namespaceType ExceptionList namespace_type
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
exports.getExceptionFilterFromExceptions = getExceptionFilterFromExceptions;
const duplicateExceptionList = async ({
  http,
  includeExpiredExceptions,
  listId,
  namespaceType,
  signal
}) => http.fetch(`${_securitysolutionListConstants.EXCEPTION_LIST_URL}/_duplicate`, {
  method: 'POST',
  query: {
    list_id: listId,
    namespace_type: namespaceType,
    include_expired_exceptions: includeExpiredExceptions
  },
  signal
});
exports.duplicateExceptionList = duplicateExceptionList;