"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateRiskScores = void 0;
var _technical_rule_data_field_names = require("@kbn/rule-registry-plugin/common/technical_rule_data_field_names");
var _with_security_span = require("../../utils/with_security_span");
var _helpers = require("./helpers");
var _risk_weights = require("./risk_weights");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bucketToResponse = ({
  bucket,
  now,
  identifierField
}) => ({
  '@timestamp': now,
  identifierField,
  identifierValue: bucket.key[identifierField],
  level: bucket.risk_details.value.level,
  totalScore: bucket.risk_details.value.score,
  totalScoreNormalized: bucket.risk_details.value.normalized_score,
  alertsScore: bucket.risk_details.value.alerts_score,
  otherScore: bucket.risk_details.value.other_score,
  notes: bucket.risk_details.value.notes,
  riskiestInputs: bucket.riskiest_inputs.hits.hits.map(riskInput => {
    var _riskInput$sort$, _riskInput$sort;
    return {
      id: riskInput._id,
      index: riskInput._index,
      riskScore: (_riskInput$sort$ = (_riskInput$sort = riskInput.sort) === null || _riskInput$sort === void 0 ? void 0 : _riskInput$sort[0]) !== null && _riskInput$sort$ !== void 0 ? _riskInput$sort$ : undefined
    };
  })
});
const filterFromRange = range => ({
  range: {
    '@timestamp': {
      lt: range.end,
      gte: range.start
    }
  }
});
const buildReduceScript = ({
  globalIdentifierTypeWeight
}) => {
  return `
    Map results = new HashMap();
    List inputs = [];
    for (state in states) {
      inputs.addAll(state.inputs)
    }
    Collections.sort(inputs, (a, b) -> b.get('weighted_score').compareTo(a.get('weighted_score')));

    double num_inputs_to_score = Math.min(inputs.length, params.max_risk_inputs_per_identity);
    results['notes'] = [];
    if (num_inputs_to_score == params.max_risk_inputs_per_identity) {
      results['notes'].add('Number of risk inputs (' + inputs.length + ') exceeded the maximum allowed (' + params.max_risk_inputs_per_identity + ').');
    }

    ${(0, _risk_weights.buildCategoryScoreDeclarations)()}

    double total_score = 0;
    double current_score = 0;
    for (int i = 0; i < num_inputs_to_score; i++) {
      current_score = inputs[i].weighted_score / Math.pow(i + 1, params.p);

      ${(0, _risk_weights.buildCategoryScoreAssignment)()}
      total_score += current_score;
    }

    ${globalIdentifierTypeWeight != null ? `total_score *= ${globalIdentifierTypeWeight};` : ''}
    double score_norm = 100 * total_score / params.risk_cap;
    results['score'] = total_score;
    results['normalized_score'] = score_norm;

    if (score_norm < 20) {
      results['level'] = 'Unknown'
    }
    else if (score_norm >= 20 && score_norm < 40) {
      results['level'] = 'Low'
    }
    else if (score_norm >= 40 && score_norm < 70) {
      results['level'] = 'Moderate'
    }
    else if (score_norm >= 70 && score_norm < 90) {
      results['level'] = 'High'
    }
    else if (score_norm >= 90) {
      results['level'] = 'Critical'
    }

    return results;
  `;
};
const buildIdentifierTypeAggregation = ({
  afterKeys,
  identifierType,
  pageSize,
  weights
}) => {
  const globalIdentifierTypeWeight = (0, _risk_weights.getGlobalWeightForIdentifierType)({
    identifierType,
    weights
  });
  const identifierField = (0, _helpers.getFieldForIdentifierAgg)(identifierType);
  return {
    composite: {
      size: pageSize,
      sources: [{
        [identifierField]: {
          terms: {
            field: identifierField
          }
        }
      }],
      after: (0, _helpers.getAfterKeyForIdentifierType)({
        identifierType,
        afterKeys
      })
    },
    aggs: {
      riskiest_inputs: {
        top_hits: {
          size: 10,
          sort: {
            [_technical_rule_data_field_names.ALERT_RISK_SCORE]: 'desc'
          },
          _source: false
        }
      },
      risk_details: {
        scripted_metric: {
          init_script: 'state.inputs = []',
          map_script: `
              Map fields = new HashMap();
              String category = doc['${_technical_rule_data_field_names.EVENT_KIND}'].value;
              double score = doc['${_technical_rule_data_field_names.ALERT_RISK_SCORE}'].value;
              double weighted_score = 0.0;

              fields.put('time', doc['@timestamp'].value);
              fields.put('category', category);
              fields.put('score', score);
              ${(0, _risk_weights.buildWeightingOfScoreByCategory)({
            userWeights: weights,
            identifierType
          })}
              fields.put('weighted_score', weighted_score);

              state.inputs.add(fields);
            `,
          combine_script: 'return state;',
          params: {
            max_risk_inputs_per_identity: 999999,
            p: 1.5,
            risk_cap: 261.2
          },
          reduce_script: buildReduceScript({
            globalIdentifierTypeWeight
          })
        }
      }
    }
  };
};
const calculateRiskScores = async ({
  afterKeys: userAfterKeys,
  debug,
  esClient,
  filter: userFilter,
  identifierType,
  index,
  logger,
  pageSize,
  range,
  weights
}) => (0, _with_security_span.withSecuritySpan)('calculateRiskScores', async () => {
  var _response$aggregation, _response$aggregation2, _response$aggregation3, _response$aggregation4, _response$aggregation5, _response$aggregation6;
  const now = new Date().toISOString();
  const filter = [{
    exists: {
      field: _technical_rule_data_field_names.ALERT_RISK_SCORE
    }
  }, filterFromRange(range)];
  if (userFilter) {
    filter.push(userFilter);
  }
  const identifierTypes = identifierType ? [identifierType] : ['host', 'user'];
  const request = {
    size: 0,
    _source: false,
    index,
    query: {
      bool: {
        filter
      }
    },
    aggs: identifierTypes.reduce((aggs, _identifierType) => {
      aggs[_identifierType] = buildIdentifierTypeAggregation({
        afterKeys: userAfterKeys,
        identifierType: _identifierType,
        pageSize,
        weights
      });
      return aggs;
    }, {})
  };
  if (debug) {
    logger.info(`Executing Risk Score query:\n${JSON.stringify(request)}`);
  }
  const response = await esClient.search(request);
  if (debug) {
    logger.info(`Received Risk Score response:\n${JSON.stringify(response)}`);
  }
  if (response.aggregations == null) {
    return {
      ...(debug ? {
        request,
        response
      } : {}),
      after_keys: {},
      scores: []
    };
  }
  const userBuckets = (_response$aggregation = (_response$aggregation2 = response.aggregations.user) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.buckets) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
  const hostBuckets = (_response$aggregation3 = (_response$aggregation4 = response.aggregations.host) === null || _response$aggregation4 === void 0 ? void 0 : _response$aggregation4.buckets) !== null && _response$aggregation3 !== void 0 ? _response$aggregation3 : [];
  const afterKeys = {
    host: (_response$aggregation5 = response.aggregations.host) === null || _response$aggregation5 === void 0 ? void 0 : _response$aggregation5.after_key,
    user: (_response$aggregation6 = response.aggregations.user) === null || _response$aggregation6 === void 0 ? void 0 : _response$aggregation6.after_key
  };
  const scores = userBuckets.map(bucket => bucketToResponse({
    bucket,
    identifierField: 'user.name',
    now
  })).concat(hostBuckets.map(bucket => bucketToResponse({
    bucket,
    identifierField: 'host.name',
    now
  })));
  return {
    ...(debug ? {
      request,
      response
    } : {}),
    after_keys: afterKeys,
    scores
  };
});
exports.calculateRiskScores = calculateRiskScores;