"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildFieldsKeyAsArrayMap = buildFieldsKeyAsArrayMap;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isObjectTypeGuard = value => {
  return (0, _lodash.isPlainObject)(value);
};
function traverseSource(document, result = {}, prefix = []) {
  if (prefix.length) {
    result[prefix.join('.')] = prefix;
  }
  if (isObjectTypeGuard(document)) {
    for (const [key, value] of Object.entries(document)) {
      const path = [...prefix, key];
      traverseSource(value, result, path);
    }
  } else if ((0, _lodash.isArray)(document)) {
    // for array of primitive values we can call traverseSource once
    if ((0, _lodash.isPlainObject)(document[0])) {
      traverseSource(document[0], result, prefix);
    } else {
      document.forEach(doc => {
        traverseSource(doc, result, prefix);
      });
    }
  }
  return result;
}

/**
 * takes object document and creates map of string field keys to array field keys
 * source  `{ 'a.b': { c: { d: 1 } } }`
 * will result in map: `{
 *     'a.b': ['a.b'],
 *     'a.b.c': ['a.b', 'c'],
 *     'a.b.c.d': ['a.b', 'c', 'd'],
 *   }`
 * @param document - Record<string, SearchTypes>
 **/
function buildFieldsKeyAsArrayMap(document) {
  if (!(0, _lodash.isPlainObject)(document)) {
    return {};
  }
  return traverseSource(document);
}