"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.splitShouldClauses = exports.filterThreatMapping = exports.createInnerAndClauses = exports.createAndOrClauses = exports.buildThreatMappingFilter = exports.buildEntriesMappingFilter = exports.MAX_CHUNK_SIZE = void 0;
var _get = _interopRequireDefault(require("lodash/fp/get"));
var _types = require("./types");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_CHUNK_SIZE = 1024;
exports.MAX_CHUNK_SIZE = MAX_CHUNK_SIZE;
const buildThreatMappingFilter = ({
  threatMapping,
  threatList,
  chunkSize,
  entryKey = 'value',
  allowedFieldsForTermsQuery
}) => {
  const computedChunkSize = chunkSize !== null && chunkSize !== void 0 ? chunkSize : MAX_CHUNK_SIZE;
  if (computedChunkSize > 1024) {
    throw new TypeError('chunk sizes cannot exceed 1024 in size');
  }
  const query = buildEntriesMappingFilter({
    threatMapping,
    threatList,
    chunkSize: computedChunkSize,
    entryKey,
    allowedFieldsForTermsQuery
  });
  const filterChunk = {
    meta: {
      alias: null,
      negate: false,
      disabled: false
    },
    query
  };
  return filterChunk;
};

/**
 * Filters out any combined "AND" entries which do not include all the threat list items.
 */
exports.buildThreatMappingFilter = buildThreatMappingFilter;
const filterThreatMapping = ({
  threatMapping,
  threatListItem,
  entryKey
}) => threatMapping.map(threatMap => {
  const atLeastOneItemMissingInThreatList = threatMap.entries.some(entry => {
    const itemValue = (0, _get.default)(entry[entryKey], threatListItem.fields);
    return itemValue == null || itemValue.length !== 1;
  });
  if (atLeastOneItemMissingInThreatList) {
    return {
      ...threatMap,
      entries: []
    };
  } else {
    return {
      ...threatMap,
      entries: threatMap.entries
    };
  }
}).filter(threatMap => threatMap.entries.length !== 0);
exports.filterThreatMapping = filterThreatMapping;
const createInnerAndClauses = ({
  threatMappingEntries,
  threatListItem,
  entryKey
}) => {
  return threatMappingEntries.reduce((accum, threatMappingEntry) => {
    const value = (0, _get.default)(threatMappingEntry[entryKey], threatListItem.fields);
    if (value != null && value.length === 1) {
      // These values could be potentially 10k+ large so mutating the array intentionally
      accum.push({
        bool: {
          should: [{
            match: {
              [threatMappingEntry[entryKey === 'field' ? 'value' : 'field']]: {
                query: value[0],
                _name: (0, _utils.encodeThreatMatchNamedQuery)({
                  id: threatListItem._id,
                  index: threatListItem._index,
                  field: threatMappingEntry.field,
                  value: threatMappingEntry.value,
                  queryType: _types.ThreatMatchQueryType.match
                })
              }
            }
          }],
          minimum_should_match: 1
        }
      });
    }
    return accum;
  }, []);
};
exports.createInnerAndClauses = createInnerAndClauses;
const createAndOrClauses = ({
  threatMapping,
  threatListItem,
  entryKey
}) => {
  const should = threatMapping.reduce((accum, threatMap) => {
    const innerAndClauses = createInnerAndClauses({
      threatMappingEntries: threatMap.entries,
      threatListItem,
      entryKey
    });
    if (innerAndClauses.length !== 0) {
      // These values could be potentially 10k+ large so mutating the array intentionally
      accum.push({
        bool: {
          filter: innerAndClauses
        }
      });
    }
    return accum;
  }, []);
  return should;
};
exports.createAndOrClauses = createAndOrClauses;
const buildEntriesMappingFilter = ({
  threatMapping,
  threatList,
  chunkSize,
  entryKey,
  allowedFieldsForTermsQuery
}) => {
  const allFieldAllowedForTermQuery = entries => entries.every(entry => {
    var _allowedFieldsForTerm, _allowedFieldsForTerm2;
    return (allowedFieldsForTermsQuery === null || allowedFieldsForTermsQuery === void 0 ? void 0 : (_allowedFieldsForTerm = allowedFieldsForTermsQuery.source) === null || _allowedFieldsForTerm === void 0 ? void 0 : _allowedFieldsForTerm[entry.field]) && (allowedFieldsForTermsQuery === null || allowedFieldsForTermsQuery === void 0 ? void 0 : (_allowedFieldsForTerm2 = allowedFieldsForTermsQuery.threat) === null || _allowedFieldsForTerm2 === void 0 ? void 0 : _allowedFieldsForTerm2[entry.value]);
  });
  const combinedShould = threatMapping.reduce((acc, threatMap) => {
    if (threatMap.entries.length > 1 || !allFieldAllowedForTermQuery(threatMap.entries)) {
      threatList.forEach(threatListSearchItem => {
        const filteredEntries = filterThreatMapping({
          threatMapping: [threatMap],
          threatListItem: threatListSearchItem,
          entryKey
        });
        const queryWithAndOrClause = createAndOrClauses({
          threatMapping: filteredEntries,
          threatListItem: threatListSearchItem,
          entryKey
        });
        if (queryWithAndOrClause.length !== 0) {
          // These values can be 10k+ large, so using a push here for performance
          acc.match.push(...queryWithAndOrClause);
        }
      });
    } else {
      const threatMappingEntry = threatMap.entries[0];
      const threats = threatList.map(threatListItem => (0, _get.default)(threatMappingEntry[entryKey], threatListItem.fields)).filter(val => val).map(val => val[0]);
      if (threats.length > 0) {
        acc.term.push({
          terms: {
            _name: (0, _utils.encodeThreatMatchNamedQuery)({
              field: threatMappingEntry.field,
              value: threatMappingEntry.value,
              queryType: _types.ThreatMatchQueryType.term
            }),
            [threatMappingEntry[entryKey === 'field' ? 'value' : 'field']]: threats
          }
        });
      }
    }
    return acc;
  }, {
    match: [],
    term: []
  });
  const matchShould = splitShouldClauses({
    should: combinedShould.match.length > 0 ? [{
      bool: {
        should: combinedShould.match,
        minimum_should_match: 1
      }
    }] : [],
    chunkSize
  });
  return {
    bool: {
      should: [...matchShould, ...combinedShould.term],
      minimum_should_match: 1
    }
  };
};
exports.buildEntriesMappingFilter = buildEntriesMappingFilter;
const splitShouldClauses = ({
  should,
  chunkSize
}) => {
  if (should.length <= chunkSize) {
    return should;
  } else {
    return should.reduce((accum, item, index) => {
      var _accum$chunkIndex$boo;
      const chunkIndex = Math.floor(index / chunkSize);
      const currentChunk = accum[chunkIndex];
      if (!currentChunk) {
        // create a new element in the array at the correct spot
        accum[chunkIndex] = {
          bool: {
            should: [],
            minimum_should_match: 1
          }
        };
      }
      // Add to the existing array element. Using mutatious push here since these arrays can get very large such as 10k+ and this is going to be a hot code spot.
      if (Array.isArray((_accum$chunkIndex$boo = accum[chunkIndex].bool) === null || _accum$chunkIndex$boo === void 0 ? void 0 : _accum$chunkIndex$boo.should)) {
        var _accum$chunkIndex$boo2;
        ((_accum$chunkIndex$boo2 = accum[chunkIndex].bool) === null || _accum$chunkIndex$boo2 === void 0 ? void 0 : _accum$chunkIndex$boo2.should).push(item);
      }
      return accum;
    }, []);
  }
};
exports.splitShouldClauses = splitShouldClauses;