"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createInstalledIntegrationSet = void 0;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createInstalledIntegrationSet = () => {
  const packageMap = new Map([]);
  const addPackage = fleetPackage => {
    var _fleetPackage$savedOb, _fleetPackage$policy_;
    if (fleetPackage.type !== 'integration') {
      return;
    }
    if (fleetPackage.status !== 'installed') {
      return;
    }
    const packageKey = `${fleetPackage.name}`;
    const existingPackageInfo = packageMap.get(packageKey);
    if (existingPackageInfo != null) {
      return;
    }

    // Actual `installed_version` is buried in SO, root `version` is latest package version available
    const installedPackageVersion = ((_fleetPackage$savedOb = fleetPackage.savedObject) === null || _fleetPackage$savedOb === void 0 ? void 0 : _fleetPackage$savedOb.attributes.install_version) || '';

    // Policy templates correspond to package's integrations.
    const packagePolicyTemplates = (_fleetPackage$policy_ = fleetPackage.policy_templates) !== null && _fleetPackage$policy_ !== void 0 ? _fleetPackage$policy_ : [];
    const packageInfo = {
      package_name: fleetPackage.name,
      package_title: fleetPackage.title,
      package_version: installedPackageVersion,
      integrations: new Map(packagePolicyTemplates.map(pt => {
        const integrationTitle = packagePolicyTemplates.length === 1 && pt.name === fleetPackage.name ? fleetPackage.title : pt.title;
        const integrationInfo = {
          integration_name: pt.name,
          integration_title: integrationTitle,
          is_enabled: false // There might not be an integration policy, so default false and later update in addPackagePolicy()
        };

        return [integrationInfo.integration_name, integrationInfo];
      }))
    };
    packageMap.set(packageKey, packageInfo);
  };
  const addPackagePolicy = policy => {
    const packageInfo = getPackageInfoFromPolicy(policy);
    const integrationsInfo = getIntegrationsInfoFromPolicy(policy, packageInfo);
    const packageKey = `${packageInfo.package_name}`;
    const existingPackageInfo = packageMap.get(packageKey);
    if (existingPackageInfo == null) {
      const integrationsMap = new Map();
      integrationsInfo.forEach(integration => {
        addIntegrationToMap(integrationsMap, integration);
      });
      packageMap.set(packageKey, {
        ...packageInfo,
        integrations: integrationsMap
      });
    } else {
      integrationsInfo.forEach(integration => {
        addIntegrationToMap(existingPackageInfo.integrations, integration);
      });
    }
  };
  const getPackages = () => {
    const packages = Array.from(packageMap.values());
    return packages.map(packageInfo => {
      const integrations = Array.from(packageInfo.integrations.values());
      return {
        ...packageInfo,
        integrations
      };
    });
  };
  const getIntegrations = () => {
    const packages = Array.from(packageMap.values());
    return (0, _lodash.flatten)(packages.map(packageInfo => {
      const integrations = Array.from(packageInfo.integrations.values());
      return integrations.map(integrationInfo => {
        return packageInfo.package_name === integrationInfo.integration_name ? {
          package_name: packageInfo.package_name,
          package_title: packageInfo.package_title,
          package_version: packageInfo.package_version,
          is_enabled: integrationInfo.is_enabled
        } : {
          package_name: packageInfo.package_name,
          package_title: packageInfo.package_title,
          package_version: packageInfo.package_version,
          integration_name: integrationInfo.integration_name,
          integration_title: integrationInfo.integration_title,
          is_enabled: integrationInfo.is_enabled
        };
      });
    }));
  };
  return {
    addPackage,
    addPackagePolicy,
    getPackages,
    getIntegrations
  };
};
exports.createInstalledIntegrationSet = createInstalledIntegrationSet;
const getPackageInfoFromPolicy = policy => {
  var _policy$package, _policy$package2, _policy$package3;
  return {
    package_name: normalizeString((_policy$package = policy.package) === null || _policy$package === void 0 ? void 0 : _policy$package.name),
    package_title: normalizeString((_policy$package2 = policy.package) === null || _policy$package2 === void 0 ? void 0 : _policy$package2.title),
    package_version: normalizeString((_policy$package3 = policy.package) === null || _policy$package3 === void 0 ? void 0 : _policy$package3.version)
  };
};
const getIntegrationsInfoFromPolicy = (policy, packageInfo) => {
  // Construct integration info from the available policy_templates
  const integrationInfos = policy.inputs.map(input => {
    var _input$policy_templat;
    const integrationName = normalizeString((_input$policy_templat = input.policy_template) !== null && _input$policy_templat !== void 0 ? _input$policy_templat : input.type); // e.g. 'cloudtrail'
    const integrationTitle = `${packageInfo.package_title} ${(0, _lodash.capitalize)(integrationName)}`; // e.g. 'AWS Cloudtrail'
    return {
      integration_name: integrationName,
      integration_title: integrationTitle,
      is_enabled: input.enabled
    };
  });

  // Base package may not have policy template, so pull directly from `policy.package` if so
  return [...integrationInfos, ...(policy.package ? [{
    integration_name: policy.package.name,
    integration_title: policy.package.title,
    is_enabled: true // Always true if `policy.package` exists since this corresponds to the base package
  }] : [])];
};
const normalizeString = raw => {
  return (raw !== null && raw !== void 0 ? raw : '').trim();
};
const addIntegrationToMap = (map, integration) => {
  if (!map.has(integration.integration_name) || integration.is_enabled) {
    map.set(integration.integration_name, integration);
  }
};