"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ResolverTreeFetcher = ResolverTreeFetcher;
var selectors = _interopRequireWildcard(require("../selectors"));
var _ecs_safety_helpers = require("../../../../common/endpoint/models/ecs_safety_helpers");
var _resolver_tree = require("../../models/resolver_tree");
var _action = require("../data/action");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * A function that handles syncing ResolverTree data w/ the current entity ID.
 * This will make a request anytime the entityID changes (to something other than undefined.)
 * If the entity ID changes while a request is in progress, the in-progress request will be cancelled.
 * Call the returned function after each state transition.
 * This is a factory because it is stateful and keeps that state in closure.
 */
function ResolverTreeFetcher(dataAccessLayer, api) {
  let lastRequestAbortController;
  // Call this after each state change.
  // This fetches the ResolverTree for the current entityID
  // if the entityID changes while
  return async id => {
    // const id = 'alerts-page';
    const state = api.getState();
    const databaseParameters = selectors.treeParametersToFetch(state.analyzer.analyzerById[id]);
    if (selectors.treeRequestParametersToAbort(state.analyzer.analyzerById[id]) && lastRequestAbortController) {
      lastRequestAbortController.abort();
      // calling abort will cause an action to be fired
    } else if (databaseParameters !== null) {
      lastRequestAbortController = new AbortController();
      let entityIDToFetch;
      let dataSource;
      let dataSourceSchema;
      let result;
      const timeRangeFilters = selectors.timeRangeFilters(state.analyzer.analyzerById[id]);

      // Inform the state that we've made the request. Without this, the middleware will try to make the request again
      // immediately.
      api.dispatch((0, _action.appRequestedResolverData)({
        id,
        parameters: databaseParameters
      }));
      try {
        const matchingEntities = await dataAccessLayer.entities({
          _id: databaseParameters.databaseDocumentID,
          indices: databaseParameters.indices,
          signal: lastRequestAbortController.signal
        });
        if (matchingEntities.length < 1) {
          // If no entity_id could be found for the _id, bail out with a failure.
          api.dispatch((0, _action.serverFailedToReturnResolverData)({
            id,
            parameters: databaseParameters
          }));
          return;
        }
        ({
          id: entityIDToFetch,
          schema: dataSourceSchema,
          name: dataSource
        } = matchingEntities[0]);
        result = await dataAccessLayer.resolverTree({
          dataId: entityIDToFetch,
          schema: dataSourceSchema,
          timeRange: timeRangeFilters,
          indices: databaseParameters.indices,
          ancestors: (0, _resolver_tree.ancestorsRequestAmount)(dataSourceSchema),
          descendants: (0, _resolver_tree.descendantsRequestAmount)()
        });
        const resolverTree = {
          originID: entityIDToFetch,
          nodes: result
        };
        if (resolverTree.nodes.length === 0) {
          const unboundedTree = await dataAccessLayer.resolverTree({
            dataId: entityIDToFetch,
            schema: dataSourceSchema,
            indices: databaseParameters.indices,
            ancestors: (0, _resolver_tree.ancestorsRequestAmount)(dataSourceSchema),
            descendants: (0, _resolver_tree.descendantsRequestAmount)()
          });
          if (unboundedTree.length > 0) {
            const timestamps = unboundedTree.map(event => (0, _ecs_safety_helpers.firstNonNullValue)(event.data['@timestamp'])).sort();
            const oldestTimestamp = timestamps[0];
            const newestTimestamp = timestamps.slice(-1);
            api.dispatch((0, _action.serverReturnedResolverData)({
              id,
              result: {
                ...resolverTree,
                nodes: unboundedTree
              },
              dataSource,
              schema: dataSourceSchema,
              parameters: databaseParameters,
              detectedBounds: {
                from: String(oldestTimestamp),
                to: String(newestTimestamp)
              }
            }));

            // 0 results with unbounded query, fail as before
          } else {
            api.dispatch((0, _action.serverReturnedResolverData)({
              id,
              result: resolverTree,
              dataSource,
              schema: dataSourceSchema,
              parameters: databaseParameters
            }));
          }
        } else {
          api.dispatch((0, _action.serverReturnedResolverData)({
            id,
            result: resolverTree,
            dataSource,
            schema: dataSourceSchema,
            parameters: databaseParameters
          }));
        }
      } catch (error) {
        // https://developer.mozilla.org/en-US/docs/Web/API/DOMException#exception-AbortError
        if (error instanceof DOMException && error.name === 'AbortError') {
          api.dispatch((0, _action.appAbortedResolverDataRequest)({
            id,
            parameters: databaseParameters
          }));
        } else {
          api.dispatch((0, _action.serverFailedToReturnResolverData)({
            id,
            parameters: databaseParameters
          }));
        }
      }
    }
  };
}