"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.hasGrandchildren = exports.getTreeNodes = exports.getTreeNode = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Helper function to check if a node has grandchildren
 * @param statsNodes type StatsNode array
 * @param children type StatsNode array
 * @return a boolean of wheather a bode has grandchildren
 */
const hasGrandchildren = (statsNodes, children) => {
  return children.some(child => statsNodes.some(node => node.parent === child.id));
};

/**
 * Helper function to create a tree node from id, name and children
 * @param id type string
 * @param name type string
 * @param children children nodes
 * @return a node of Node type
 */
exports.hasGrandchildren = hasGrandchildren;
const getTreeNode = (id, name, children) => {
  return {
    label: name,
    id,
    isExpanded: true,
    children
  };
};

/**
 * Helper function to create tree nodes based on statsNode list from resolver api
 * @param statsNodes type StatsNode[]
 * @param childrenCountLimit optional parameter to limit the number of children displayed, default to 3
 * @return a node list for EuiTreeView
 */
exports.getTreeNode = getTreeNode;
const getTreeNodes = (statsNodes, childCountLimit = 3) => {
  if (statsNodes.length === 0) {
    return [];
  }
  const node = statsNodes[0];
  const nodeList = [];
  const currentNode = getTreeNode(node.id, `--> (Analyzed Event) ${node.name}`, []);
  const children = statsNodes.filter(item => item.parent === node.id);
  if (children && children.length !== 0) {
    children.forEach((child, idx) => {
      if (idx < childCountLimit) {
        var _currentNode$children;
        (_currentNode$children = currentNode.children) === null || _currentNode$children === void 0 ? void 0 : _currentNode$children.push(getTreeNode(child.id, `--> ${child.name}`, []));
      }
    });
  }
  const parent = statsNodes.find(item => item.id === node.parent);
  if (parent) {
    const parentNode = getTreeNode(parent.id, `--> ${parent.name}`, [currentNode]);
    if (parent !== null && parent !== void 0 && parent.parent) {
      nodeList.push(getTreeNode('grandparent', '...', [parentNode]));
    } else {
      nodeList.push(parentNode);
    }
  } else {
    nodeList.push(currentNode);
  }
  if (hasGrandchildren(statsNodes, children)) {
    nodeList.push(getTreeNode('grandchild', '...', []));
  }
  return nodeList;
};
exports.getTreeNodes = getTreeNodes;